// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.io.InputStream;
import java.net.InetSocketAddress;

import javax.xml.ws.Endpoint;

import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpServerResponse;
import org.refcodes.net.HttpStatusException;
import org.refcodes.net.RequestHeaderFields;

/**
 * Extends a {@link RestServer} to be used as loopback device e.g. for testing
 * purposes such as testing your {@link RestRequestObserver} implementations.
 * Use the
 * {@link #onHttpRequest(InetSocketAddress, InetSocketAddress, HttpMethod, String, String, RequestHeaderFields, InputStream)}
 * method to simulate REST requests on the {@link LoopbackRestServer}. An
 * registered {@link Endpoint} instances being targeted at will be invoked
 * accordingly.
 */
public interface LoopbackRestServer extends RestServer {

	/**
	 * Invoke this method to simulate a REST request to be handled by the
	 * according registered {@link Endpoint} instances:.
	 *
	 * @param aLocalAddress the a local address
	 * @param aClientAddress the a client address
	 * @param aHttpMethod The method for the simulated REST request to be used
	 * @param aLocator The locator for the simulated REST request to be used
	 *        without the query string portion.
	 * @param aQueryString The query string part of the request.
	 * @param aRequestHeaderFields The simulated REST request's Header-Fields to
	 *        be used
	 * @param aHttpBody The raw body for the simulated REST request to be used.
	 * @return The {@link HttpServerResponse} containing the complete result of
	 *         processing the request.
	 * @throws HttpStatusException the http status exception
	 */
	public HttpServerResponse onHttpRequest( InetSocketAddress aLocalAddress, InetSocketAddress aClientAddress, HttpMethod aHttpMethod, String aLocator, String aQueryString, RequestHeaderFields aRequestHeaderFields, String aHttpBody ) throws HttpStatusException;

	/**
	 * Invoke this method to simulate a REST request to be handled by the
	 * according registered {@link Endpoint} instances:.
	 *
	 * @param aLocalAddress the a local address
	 * @param aClientAddress the a client address
	 * @param aHttpMethod The method for the simulated REST request to be used
	 * @param aLocator The locator for the simulated REST request to be used
	 *        without the query string portion.
	 * @param aQueryString The query string part of the request.
	 * @param aRequestHeaderFields The simulated REST request's Header-Fields to
	 *        be used
	 * @param aHttpInputStream The HTTP {@link InputStream} representing the
	 *        body for the simulated REST request to be used.
	 * @return The {@link HttpServerResponse} containing the complete result of
	 *         processing the request.
	 * @throws HttpStatusException the http status exception
	 */
	public HttpServerResponse onHttpRequest( InetSocketAddress aLocalAddress, InetSocketAddress aClientAddress, HttpMethod aHttpMethod, String aLocator, String aQueryString, RequestHeaderFields aRequestHeaderFields, InputStream aHttpInputStream ) throws HttpStatusException;

	/**
	 * With realm.
	 *
	 * @param aRealm the a realm
	 * @return the loopback rest server
	 */
	@Override
	default LoopbackRestServer withRealm( String aRealm ) {
		setRealm( aRealm );
		return this;
	}

	/**
	 * With base locator.
	 *
	 * @param aBaseLocator the a base locator
	 * @return the loopback rest server
	 */
	@Override
	default LoopbackRestServer withBaseLocator( String aBaseLocator ) {
		setBaseLocator( aBaseLocator );
		return this;
	}
}
