// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatusAccessor;
import org.refcodes.component.LinkComponent.LinkComponentBuilder;
import org.refcodes.component.OpenException;
import org.refcodes.mixin.LocatorAccessor.LocatorBuilder;
import org.refcodes.mixin.LocatorAccessor.LocatorProperty;
import org.refcodes.net.FormFields;
import org.refcodes.net.HeaderFieldsAccessor.HeaderFieldsBuilder;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpMethodAccessor.HttpMethodBuilder;
import org.refcodes.net.HttpMethodAccessor.HttpMethodProperty;
import org.refcodes.net.HttpRequestBuilder;
import org.refcodes.net.QueryFieldsAccessor.QueryFieldsBuilder;
import org.refcodes.net.RequestHeaderFields;

/**
 * An {@link RestCallerBuilder} extends an {@link RestCaller} with builder
 * functionality and adds <code>lambda</code> support for handling the responses
 * addressed to this {@link RestCaller}. The <code>lambda</code> defined as
 * {@link RestResponseObserver} acts as the single listener to this
 * {@link RestCaller} responsible for handling the responses for which this
 * {@link RestCaller} is responsible.
 * 
 * The locator to which a {@link RestCallerBuilder} targets for is defined by
 * the {@link #getLocator()} property.
 */
public interface RestCallerBuilder extends RestCaller, HttpRequestBuilder<RestCallerBuilder>, LocatorProperty<String>, LocatorBuilder<String, RestCallerBuilder>, HttpMethodProperty, HttpMethodBuilder<RestCallerBuilder>, LinkComponentBuilder<RestCallerBuilder>, ConnectionStatusAccessor, QueryFieldsBuilder<RestCallerBuilder>, HeaderFieldsBuilder<RequestHeaderFields, RestCallerBuilder> {

	/**
	 * With query fields.
	 *
	 * @param aQueryFields the a query fields
	 * @return the rest caller builder
	 */
	@Override
	default RestCallerBuilder withQueryFields( FormFields aQueryFields ) {
		setQueryFields( aQueryFields );
		return this;
	}

	/**
	 * With locator.
	 *
	 * @param aLocator the a locator
	 * @return the rest caller builder
	 */
	@Override
	default RestCallerBuilder withLocator( String aLocator ) {
		setLocator( aLocator );
		return this;
	}

	/**
	 * With http method.
	 *
	 * @param aHttpMethod the a http method
	 * @return the rest caller builder
	 */
	@Override
	default RestCallerBuilder withHttpMethod( HttpMethod aHttpMethod ) {
		setHttpMethod( aHttpMethod );
		return this;
	}

	/**
	 * With request.
	 *
	 * @param <REQ> the generic type
	 * @param aRequest the a request
	 * @return the rest caller builder
	 */
	@Override
	default <REQ> RestCallerBuilder withRequest( REQ aRequest ) {
		setRequest( aRequest );
		return this;
	}

	/**
	 * Retrieves the {@link RestResponseObserver} to which any responses are
	 * delegated upon invocation of the {@link #onResponse(RestResponseEvent)}
	 * method.
	 * 
	 * @return The (user defined) {@link RestResponseObserver} to handle
	 *         responses.
	 */
	RestResponseObserver getResponseObserver();

	/**
	 * Sets the {@link RestResponseObserver} to which any responses are
	 * delegated upon invocation of the {@link #onResponse(RestResponseEvent)}
	 * method.
	 * 
	 * @param aLambda The (user defined) {@link RestResponseObserver} to handle
	 *        responses, feel free to code it as <code>lambda</code> expression!
	 */
	void setResponseObserver( RestResponseObserver aLambda );

	/**
	 * Builder method for setting the {@link RestResponseObserver}.
	 * 
	 * @param aLambda The (user defined) {@link RestResponseObserver} to handle
	 *        responses, feel free to code it as <code>lambda</code> expression
	 * 
	 * @return The {@link RestCallerBuilder} for the sake of a fluent API.
	 */
	default RestCallerBuilder withResponseObserver( RestResponseObserver aLambda ) {
		setResponseObserver( aLambda );
		return this;
	}

	/**
	 * With Header-Fields.
	 *
	 * @param aRequestHeaderFields the a request Header-Fields
	 * @return the rest caller builder
	 */
	@Override
	default RestCallerBuilder withHeaderFields( RequestHeaderFields aRequestHeaderFields ) {
		setHeaderFields( aRequestHeaderFields );
		return this;
	}

	/**
	 * With open.
	 *
	 * @return the rest caller builder
	 * @throws OpenException the open exception
	 */
	@Override
	default RestCallerBuilder withOpen() throws OpenException {
		open();
		return this;
	}

	/**
	 * With close.
	 *
	 * @return the rest caller builder
	 * @throws CloseException the close exception
	 */
	@Override
	default RestCallerBuilder withClose() throws CloseException {
		close();
		return this;
	}

	/**
	 * With close quietly.
	 *
	 * @return the rest caller builder
	 */
	@Override
	default RestCallerBuilder withCloseQuietly() {
		closeQuietly();
		return this;
	}

	/**
	 * With close in.
	 *
	 * @param aCloseInMillis the a close in millis
	 * @return the rest caller builder
	 */
	@Override
	default RestCallerBuilder withCloseIn( int aCloseInMillis ) {
		closeIn( aCloseInMillis );
		return this;
	}
}
