// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import java.io.InputStream;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import org.refcodes.controlflow.ExecutionStrategy;
import org.refcodes.data.Delimiter;
import org.refcodes.data.Encoding;
import org.refcodes.exception.BugException;
import org.refcodes.exception.MarshalException;
import org.refcodes.exception.VetoException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.matcher.WildcardSubstitutes;
import org.refcodes.matcher.impls.PathMatcherImpl;
import org.refcodes.net.BasicAuthRequiredException;
import org.refcodes.net.ContentType;
import org.refcodes.net.FormFields;
import org.refcodes.net.HeaderField;
import org.refcodes.net.HeaderFields;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpRequest;
import org.refcodes.net.HttpServerResponse;
import org.refcodes.net.HttpStatusException;
import org.refcodes.net.InternalServerErrorException;
import org.refcodes.net.MediaType;
import org.refcodes.net.MediaTypeFactory;
import org.refcodes.net.MediaTypeParameter;
import org.refcodes.net.NotFoundException;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.ResponseHeaderFields;
import org.refcodes.net.UnsupportedMediaTypeException;
import org.refcodes.net.impls.ApplicationFormFactory;
import org.refcodes.net.impls.ApplicationJsonFactory;
import org.refcodes.net.impls.ApplicationXmlFactory;
import org.refcodes.net.impls.ContentTypeImpl;
import org.refcodes.net.impls.FormFieldsImpl;
import org.refcodes.net.impls.HttpServerResponseImpl;
import org.refcodes.net.impls.TextPlainFactory;
import org.refcodes.observer.impls.AbstractObservable;
import org.refcodes.rest.RestEndpoint;
import org.refcodes.rest.RestEndpointBuilder;
import org.refcodes.rest.RestRequestEvent;
import org.refcodes.rest.RestRequestObserver;
import org.refcodes.rest.RestServer;
import org.refcodes.runtime.Correlation;
import org.refcodes.runtime.SystemUtility;
import org.refcodes.structure.Properties;
import org.refcodes.structure.Properties.PropertiesBuilder;
import org.refcodes.textual.impls.VerboseTextBuilderImpl;

/**
 * Implementation of the base functionality of the {@link RestServer} interface
 * omitting the HTTP handling part being the foundation for various
 * {@link RestServer} implementations such as {@link HttpRestServerImpl} or
 * {@link LoopbackRestServerImpl}.
 * 
 * The {@link AbstractRestServer} is pre-configured with the following
 * {@link MediaTypeFactory} instances:
 * 
 * <ul>
 * <li>{@link ApplicationJsonFactory}</li>
 * <li>{@link ApplicationXmlFactory}</li>
 * <li>{@link TextPlainFactory}</li>
 * <li>{@link ApplicationFormFactory}</li>
 * </ul>
 * 
 * In your sub-classes, overwrite the method {@link #initMedaTypeFactories()},
 * therein calling {@link #addMediaTypeFactory(MediaTypeFactory)} to add (by
 * also invoking super's {@link #initMedaTypeFactories()}) or to set your own
 * (without invoking super's {@link #initMedaTypeFactories()})
 * {@link MediaTypeFactory} instances.
 */
public abstract class AbstractRestServer extends AbstractObservable<RestEndpoint, HttpRequest> implements RestServer {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Map<MediaType, MediaTypeFactory> _mediaTypeFacotries = new LinkedHashMap<>();

	private Map<PathMatcherImpl, List<RestEndpoint>> _matcherEndpoints = new LinkedHashMap<>();

	private String _realm = SystemUtility.getComputerName();

	private String _baseLocator = null;

	protected boolean _hasRequestCorrelation = true;

	protected boolean _hasSessionCorrelation = true;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a {@link AbstractRestServer} pre-configured with
	 * {@link MediaTypeFactory} instances for JSON and REST.
	 */
	public AbstractRestServer() {
		initMedaTypeFactories();
	}

	/**
	 * CConstructs a {@link AbstractRestServer} pre-configured with
	 * {@link MediaTypeFactory} instances for JSON and REST.
	 * 
	 * @param aExecutorService An executor service to be used when creating
	 *        {@link Thread}s.
	 */
	public AbstractRestServer( ExecutorService aExecutorService ) {
		super( aExecutorService );
		initMedaTypeFactories();
	}

	/**
	 * Adds the default {@link MediaTypeFactory} instances. Can be overridden.
	 */
	protected void initMedaTypeFactories() {
		addMediaTypeFactory( new ApplicationJsonFactory() );
		addMediaTypeFactory( new ApplicationXmlFactory() );
		addMediaTypeFactory( new TextPlainFactory() );
		addMediaTypeFactory( new ApplicationFormFactory() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void setRequestCorrelation( boolean hasRequestCorrelation ) {
		_hasRequestCorrelation = hasRequestCorrelation;
	}

	@Override
	public boolean hasRequestCorrelation() {
		return _hasRequestCorrelation;
	}

	@Override
	public void setSessionCorrelation( boolean hasSessionCorrelation ) {
		_hasSessionCorrelation = hasSessionCorrelation;
	}

	@Override
	public boolean hasSessionCorrelation() {
		return _hasSessionCorrelation;
	}

	@Override
	public String getRealm() {
		return _realm;
	}

	@Override
	public void setRealm( String aRealm ) {
		_realm = aRealm;
	}

	@Override
	public String getBaseLocator() {
		return _baseLocator;
	}

	@Override
	public void setBaseLocator( String aBaseLocator ) {
		if ( !aBaseLocator.startsWith( Delimiter.PATH.getChar() + "" ) ) {
			throw new IllegalArgumentException( "Your provided base locator <" + aBaseLocator + "> is not an absolute locator, it has to start with a slash (\"" + Delimiter.PATH.getChar() + "\") character to be an absolute locator." );
		}
		_baseLocator = aBaseLocator;
	}

	@Override
	public synchronized boolean subscribeObserver( RestEndpoint aObserver ) {
		if ( super.subscribeObserver( aObserver ) ) {
			PathMatcherImpl theMatcher = new PathMatcherImpl( aObserver.getLocatorPattern() );
			List<RestEndpoint> theEndpoints = _matcherEndpoints.get( theMatcher );
			if ( theEndpoints == null ) {
				theEndpoints = new ArrayList<>();
				_matcherEndpoints.put( theMatcher, theEndpoints );
			}
			theEndpoints.add( aObserver );
			return true;
		}
		return false;
	}

	@Override
	public RestEndpointBuilder onRequest( HttpMethod aHttpMethod, String aLocatorPattern, RestRequestObserver aRequestObserver ) {
		RestEndpointBuilder theEndpoint = new RestEndpointBuilderImpl( aHttpMethod, aLocatorPattern, aRequestObserver );
		if ( !subscribeObserver( theEndpoint ) ) {
			throw new BugException( "We encountered a bug! As we created the endpoint within this method, it cannot have been added already!" );
		}
		return theEndpoint;
	}

	@Override
	public synchronized boolean unsubscribeObserver( RestEndpoint aObserver ) {
		if ( super.unsubscribeObserver( aObserver ) ) {
			Iterator<PathMatcherImpl> eMatchers = _matcherEndpoints.keySet().iterator();
			List<RestEndpoint> eObservers;
			Iterator<RestEndpoint> eEndpoints;
			while ( eMatchers.hasNext() ) {
				eObservers = _matcherEndpoints.get( eMatchers.next() );
				// if ( eObservers != null ) { // To be resilent ..
				eEndpoints = eObservers.iterator();
				while ( eEndpoints.hasNext() ) {
					if ( aObserver == eEndpoints.next() ) {
						eEndpoints.remove();
					}
				}
				if ( eObservers.isEmpty() ) {
					eMatchers.remove();
				}
				// } // More to do here in case of resilence ...
			}
			return true;
		}
		return false;
	}

	@Override
	public synchronized boolean addMediaTypeFactory( MediaTypeFactory aMediaTypeFactory ) {
		if ( _mediaTypeFacotries.containsKey( aMediaTypeFactory.getMediaType() ) ) return false;
		_mediaTypeFacotries.put( aMediaTypeFactory.getMediaType(), aMediaTypeFactory );
		return true;
	}

	@Override
	public MediaTypeFactory toMediaTypeFactory( MediaType aMediaType ) {
		return _mediaTypeFacotries.get( aMediaType );
	}

	@Override
	public MediaType[] getFactoryMediaTypes() {
		return _mediaTypeFacotries.keySet().toArray( new MediaType[_mediaTypeFacotries.size()] );
	}

	@Override
	public void dispose() {
		_matcherEndpoints.clear();
		super.dispose();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Extensions of this class disect an incoming request and pass it to this
	 * method for doing the actual invocation of the registered
	 * {@link RestEndpoint} instances. An extension might call this method from
	 * inside an event (request) handler.
	 * 
	 * @param aLocalAddress The host and port of your REST service.
	 * @param aRemoteAddress The host and port for the caller.
	 * @param aHttpMethod The {@link HttpMethod} of the request.
	 * @param aLocator The locator (URL) targeted by the request (without the
	 *        query string).
	 * @param aQueryString The query string part of the request.
	 * @param aRequestHeaderFields The Header-Fields ({@link HeaderFields})
	 *        belonging to the request.
	 * @param aHttpInputStream The body passed by the request.
	 * @return A {@link HttpServerResponse} instance to by used by the extension
	 *         to produce an according HTTP-Response.
	 * @throws HttpStatusException thrown in case of an {@link RestEndpoint}
	 *         responsible for the given request encountered a problem or none
	 *         {@link RestEndpoint} felt responsible to produce a
	 *         {@link HttpServerResponse}.
	 */
	protected HttpServerResponse onHttpRequest( InetSocketAddress aLocalAddress, InetSocketAddress aRemoteAddress, HttpMethod aHttpMethod, String aLocator, String aQueryString, RequestHeaderFields aRequestHeaderFields, InputStream aHttpInputStream ) throws HttpStatusException {
		if ( _baseLocator == null || aLocator.toLowerCase().startsWith( _baseLocator.toLowerCase() ) ) {
			if ( _baseLocator != null ) {
				aLocator = aLocator.substring( _baseLocator.length() );
			}
			RestRequestEvent eRestRequestEvent;
			HttpServerResponse theServerResponse = null;
			Object eResponse = null;
			List<RestEndpoint> eEndpoints;
			FormFields eFormFields;
			RestEndpoint theEndpoint = null;
			WildcardSubstitutes eWildcardSubstitutes;
			for ( PathMatcherImpl eMatcher : _matcherEndpoints.keySet() ) {
				eWildcardSubstitutes = eMatcher.toWildcardSubstitutes( aLocator );
				if ( eWildcardSubstitutes != null ) {
					eEndpoints = _matcherEndpoints.get( eMatcher );
					for ( RestEndpoint eEndpoint : eEndpoints ) {
						if ( eEndpoint.getHttpMethod() == null || eEndpoint.getHttpMethod() == aHttpMethod ) {
							if ( theServerResponse == null ) {
								theServerResponse = new HttpServerResponseImpl( this );
								ContentType theMediaType = toNegotiatedContenType( aRequestHeaderFields );
								if ( theMediaType != null ) {
									theServerResponse.getHeaderFields().putContentType( theMediaType );
								}
								theEndpoint = eEndpoint;
							}
							eFormFields = new FormFieldsImpl();
							eFormFields.fromHttpQueryString( aQueryString );
							eRestRequestEvent = new RestRequestEventImpl( aLocalAddress, aRemoteAddress, aHttpMethod, aLocator, eWildcardSubstitutes, eFormFields, aRequestHeaderFields, aHttpInputStream, this );

							doRequestCorrelation( aRequestHeaderFields, theServerResponse );
							doSessionCorrelation( aRequestHeaderFields, theServerResponse );

							try {
								eEndpoint.onRequest( eRestRequestEvent, theServerResponse );
							}
							catch ( BasicAuthRequiredException e ) {
								theServerResponse.getHeaderFields().putBasicAuthRequired( getRealm() );
								throw e;
							}
							if ( eResponse == null ) {
								eResponse = theServerResponse.getResponse();
							}
							else if ( eResponse != null ) {
								LOGGER.warn( "An endpoint of type <" + eEndpoint.getClass().getName() + "> (" + eEndpoint + ") would overwrite the response already produced by an endpoint of type <" + theEndpoint.getClass().getName() + "> (" + theEndpoint + " )" );
								throw new InternalServerErrorException( "Unambiguous responsibility detected for handling resource locator <" + aLocator + "> with HTTP-Method <" + aHttpMethod + ">." );
							}
						}
					}
				}
			}
			if ( theServerResponse != null ) return theServerResponse;
		}
		throw new NotFoundException( "There is none endpoint for handling resource locator <" + aLocator + "> with HTTP-Method <" + aHttpMethod + ">." );
	}

	/**
	 * Determines the best fitting respone's {@link ContentType}. The default
	 * Content-Type-Negotiation implementation of this method makes use of the
	 * {@link RequestHeaderFields} and matches them against the supported
	 * {@link MediaType} types ( retrieved via {@link #getFactoryMediaTypes()}).
	 * 
	 * May be overwritten to enforce another Content-Type-Negotiation strategy.
	 * 
	 * @param aRequestHeaderFields The request's {@link HeaderField} instance to
	 *        use when determining the best fitting respone's
	 *        {@link ContentType}.
	 * 
	 * @return The best fitting (as of the implemented Content-Type-Negotiation
	 *         strategy) Content-Type to be used for the response.
	 */
	protected ContentType toNegotiatedContenType( RequestHeaderFields aRequestHeaderFields ) {
		ContentType theMediatype = null;
		out: {
			List<ContentType> theRequestAcceptTypes = aRequestHeaderFields.getAcceptTypes();
			if ( theRequestAcceptTypes != null ) {
				for ( ContentType theContentType : theRequestAcceptTypes ) {
					if ( hasMediaTypeFactory( theContentType.getMediaType() ) ) {
						theMediatype = theContentType;
						break out;
					}
				}
			}

			// |--> Any unknown Accept-Types?
			List<String> theUnkonwnAcceptTypes = aRequestHeaderFields.getUnknownAcceptTypes();
			if ( theUnkonwnAcceptTypes != null && theUnkonwnAcceptTypes.size() != 0 ) {
				LOGGER.warn( "Unable to resolve unknown request's Header-Field <" + HeaderField.ACCEPT.getName() + ">: " + new VerboseTextBuilderImpl().withElements( theUnkonwnAcceptTypes ).toString() );
			}
			// Any unknown Accept-Types? <--|

			ContentType theRequestContentType = aRequestHeaderFields.getContentType();
			if ( theRequestContentType != null && hasMediaTypeFactory( theRequestContentType.getMediaType() ) ) {
				theMediatype = theRequestContentType;
			}
			// |--> Any unknown Content-Types?
			List<String> theUnkonwnContentTypes = aRequestHeaderFields.getUnknownContentTypes();
			if ( theUnkonwnContentTypes != null && theUnkonwnContentTypes.size() != 0 ) {
				LOGGER.warn( "Unable to resolve unknown request's Header-Field <" + HeaderField.CONTENT_TYPE.getName() + ">: " + new VerboseTextBuilderImpl().withElements( theUnkonwnContentTypes ).toString() );
			}
			// Any unknown Content-Types? <--|
		}
		return theMediatype;
	}

	/**
	 * Creates a {@link String} {@link MediaType} encoded as of the
	 * {@link HeaderField#CONTENT_TYPE} from the response header or if not set
	 * as of the {@link HeaderField#ACCEPT} from the request header or if not
	 * set as of the {@link HeaderField#CONTENT_TYPE} from the request header.
	 * 
	 * @param aResponse The response which to encode as of the detected
	 *        {@link MediaType}s.
	 * @param aRequestHeaderFields The Header-Fields from the request.
	 * @param aResponseHeaderFields The Header-Fields from the response.
	 * @return An accordingly encoded response as byte array.
	 * 
	 * @throws MarshalException thrown when marshaling / serializing an object
	 *         failed.
	 * @throws UnsupportedMediaTypeException thrown in case none of the
	 *         identified media types is supported, e.g. no required
	 *         {@link MediaTypeFactory} has been registered as of
	 *         {@link #addMediaTypeFactory(MediaTypeFactory)}.
	 */
	protected byte[] toResponseBody( Object aResponse, RequestHeaderFields aRequestHeaderFields, ResponseHeaderFields aResponseHeaderFields ) throws MarshalException, UnsupportedMediaTypeException {

		if ( aResponse == null ) return new byte[] {};

		// 1. Response Content-Type:
		String theResponseBody = toMarshaled( aResponse, aResponseHeaderFields.getContentType(), aResponseHeaderFields );
		if ( theResponseBody != null ) {
			LOGGER.info( "Auto-determined Response-Header's <" + HeaderField.CONTENT_TYPE.getName() + "> Media-Type <" + aResponseHeaderFields.getContentType().toHttpMediaType() + "> for the response." );
			return theResponseBody.getBytes();
		}
		// 2. Request Accept-Types:
		List<ContentType> theAcceptTypes = aRequestHeaderFields.getAcceptTypes();
		if ( theAcceptTypes != null ) {
			for ( ContentType eContentType : theAcceptTypes ) {
				theResponseBody = toMarshaled( aResponse, eContentType, aResponseHeaderFields );
				if ( theResponseBody != null ) {
					LOGGER.info( "Auto-determined Request-Header's <" + HeaderField.ACCEPT.getName() + "> Media-Type <" + eContentType.toHttpMediaType() + "> for the response." );
					return theResponseBody.getBytes();
				}
			}
		}

		// 3. Request Content-Type:
		theResponseBody = toMarshaled( aResponse, aRequestHeaderFields.getContentType(), aResponseHeaderFields );
		if ( theResponseBody != null ) {
			LOGGER.info( "Auto-determined Request-Header's <" + HeaderField.CONTENT_TYPE.getName() + "> Media-Type <" + aRequestHeaderFields.getContentType().toHttpMediaType() + "> for the response." );
			return theResponseBody.getBytes();
		}

		// 4. Some detectable Header-Field types were provided, but no factory:
		if ( aResponseHeaderFields.getContentType() != null || aRequestHeaderFields.getContentType() != null || (aRequestHeaderFields.getAcceptTypes() != null && aRequestHeaderFields.getAcceptTypes().size() != 0) ) {
			throw new UnsupportedMediaTypeException( "No Media-Type factory found for request ACCEPT types <" + new VerboseTextBuilderImpl().withElements( aRequestHeaderFields.getAcceptTypes() ) + "> or response CONTENT-TYPE <" + aResponseHeaderFields.getContentType() + "> or request CONTENT type <" + aRequestHeaderFields.getContentType() + ">." );
		}

		// 5. No Header-Field types provided, using Media-Type factories:
		MediaType[] theMediaTypes = getFactoryMediaTypes();
		if ( theMediaTypes != null && theMediaTypes.length != 0 ) {
			theResponseBody = toMarshaled( aResponse, theMediaTypes[0], Encoding.UTF_8.getCode(), aResponseHeaderFields );
			if ( theResponseBody != null ) {
				LOGGER.info( "Auto-configured first supported fallback Media-Type <" + theMediaTypes[0].toHttpMediaType() + "> for the response." );
				return theResponseBody.getBytes();
			}
		}

		// 6. No Header-Field types and no Media-Type factories found
		throw new UnsupportedMediaTypeException( "No fallback Media-Type factory for media type <" + MediaType.TEXT_PLAIN.toHttpMediaType() + "> found." );
	}

	@Override
	protected boolean fireEvent( HttpRequest aEvent, RestEndpoint aObserver, ExecutionStrategy aExecutionStrategy ) throws VetoException {
		throw new UnsupportedOperationException( "As the #onHttpRequest method takes care of observer invocation." );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Do request correlation.
	 *
	 * @param aRequestHeaderFields the a request Header-Fields
	 * @param aServerResponse the a server response
	 */
	protected void doRequestCorrelation( RequestHeaderFields aRequestHeaderFields, HttpServerResponse aServerResponse ) {
		if ( _hasRequestCorrelation ) {
			String theRequestId = (aRequestHeaderFields.getRequestId() != null) ? aRequestHeaderFields.getRequestId() : Correlation.REQUEST.nextId();
			aServerResponse.getHeaderFields().putRequestId( theRequestId );
		}
	}

	/**
	 * Do session correlation.
	 *
	 * @param aRequestHeaderFields the a request Header-Fields
	 * @param aServerResponse the a server response
	 */
	protected void doSessionCorrelation( RequestHeaderFields aRequestHeaderFields, HttpServerResponse aServerResponse ) {
		if ( _hasSessionCorrelation ) {
			String theSessionId = (aRequestHeaderFields.getSessionId() != null) ? aRequestHeaderFields.getSessionId() : Correlation.SESSION.nextId();
			aServerResponse.getHeaderFields().putSessionId( theSessionId );
		}
	}

	/**
	 * To marshaled.
	 *
	 * @param aResponse the a response
	 * @param aMediaType the a media type
	 * @param aMediaTypeParams the a media type params
	 * @param aResponseHeaderFields the a response Header-Fields
	 * @return the string
	 */
	private String toMarshaled( Object aResponse, MediaType aMediaType, Properties aMediaTypeParams, ResponseHeaderFields aResponseHeaderFields ) {
		if ( aMediaTypeParams != null && aMediaTypeParams.isEmpty() ) aMediaTypeParams = null;
		MediaTypeFactory theFactory = toMediaTypeFactory( aMediaType );
		if ( theFactory != null ) {
			try {
				String theMarshaled = theFactory.toMarshaled( aResponse, aMediaTypeParams );
				ContentType theContentType = new ContentTypeImpl( aMediaType, aMediaTypeParams );
				aResponseHeaderFields.putContentType( theContentType );
				return theMarshaled;
			}
			catch ( Exception e ) {}
		}
		return null;
	}

	/**
	 * To marshaled.
	 *
	 * @param aResponse the a response
	 * @param contentType the content type
	 * @param aResponseHeaderFields the a response Header-Fields
	 * @return the string
	 */
	private String toMarshaled( Object aResponse, ContentType contentType, ResponseHeaderFields aResponseHeaderFields ) {
		return toMarshaled( aResponse, contentType != null ? contentType.getMediaType() : null, contentType, aResponseHeaderFields );
	}

	/**
	 * To marshaled.
	 *
	 * @param aResponse the a response
	 * @param aMediaType the a media type
	 * @param aCharset the a charset
	 * @param aResponseHeaderFields the a response Header-Fields
	 * @return the string
	 */
	private String toMarshaled( Object aResponse, MediaType aMediaType, String aCharset, ResponseHeaderFields aResponseHeaderFields ) {
		if ( aCharset == null ) {
			return toMarshaled( aResponse, aMediaType, (Properties) null, aResponseHeaderFields );
		}
		return toMarshaled( aResponse, aMediaType, PropertiesBuilder.toPropertiesBuilder().withPut( MediaTypeParameter.CHARSET.getName(), aCharset ), aResponseHeaderFields );
	}
}
