// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.data.Delimiter;
import org.refcodes.data.Protocol;
import org.refcodes.data.SystemProperty;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.BadRequestException;
import org.refcodes.net.ContentType;
import org.refcodes.net.HttpClientRequest;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.HttpStatusCode;
import org.refcodes.net.MediaTypeFactory;
import org.refcodes.net.Port;
import org.refcodes.net.ResponseHeaderFields;
import org.refcodes.net.impls.ApplicationFormFactory;
import org.refcodes.net.impls.ApplicationJsonFactory;
import org.refcodes.net.impls.ApplicationXmlFactory;
import org.refcodes.net.impls.ResponseHeaderFieldsImpl;
import org.refcodes.net.impls.TextPlainFactory;
import org.refcodes.rest.HttpRestClient;
import org.refcodes.rest.RestRequestHandler;
import org.refcodes.rest.RestResponse;

/**
 * The {@link HttpRestClientImpl} implements the {@link HttpRestClient}
 * interface.
 * 
 * The {@link HttpRestClientImpl} is being initialized with some common
 * {@link MediaTypeFactory} instances (as implemented by the
 * {@link AbstractRestClient}). At the time of writing this document the
 * {@link MediaTypeFactory} instances being pre-configured are:
 * 
 * <ul>
 * <li>{@link ApplicationJsonFactory}</li>
 * <li>{@link ApplicationXmlFactory}</li>
 * <li>{@link TextPlainFactory}</li>
 * <li>{@link ApplicationFormFactory}</li>
 * </ul>
 * 
 * The {@link HttpRestClientImpl} supports HTTP as well as HTTPS protocols as
 * being based on the {@link HttpURLConnection}.
 * 
 * For configuring HTTPS capabilities, refer to the methods such as
 * {@link #open(File, String, String)} or {@link #open(File, String)}.
 */
public class HttpRestClientImpl extends AbstractRestClient implements HttpRestClient {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int PIPE_STREAM_BUFFER = 1024;

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private URL _baseUrl = null;

	private ConnectionStatus _connectionStatus = ConnectionStatus.NONE;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new http rest client impl.
	 */
	public HttpRestClientImpl() {
		super();
		onRestRequest( new HttpRestRequestHandler() );
	}

	/**
	 * Instantiates a new http rest client impl.
	 *
	 * @param aExecutorService the a executor service
	 */
	public HttpRestClientImpl( ExecutorService aExecutorService ) {
		super( aExecutorService );
		onRestRequest( new HttpRestRequestHandler() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void open() throws OpenException {
		if ( _connectionStatus == ConnectionStatus.OPENED ) throw new OpenException( "The HTTP rest client is already open, close first before opening again!" );
		_connectionStatus = ConnectionStatus.OPENED;
	}

	@Override
	public void open( File aKeyStoreFile, String aKeyStoreType, String aKeyStorePassword ) throws OpenException {
		SystemProperty.TRUST_STORE_FILE.setValue( aKeyStoreFile.getAbsolutePath() );
		SystemProperty.TRUST_STORE_PASSWORD.setValue( aKeyStorePassword );
		SystemProperty.TRUST_STORE_TYPE.setValue( aKeyStoreType );
	}

	@Override
	public void close() throws CloseException {
		_connectionStatus = ConnectionStatus.CLOSED;
	}

	@Override
	public URL getBaseUrl() {
		return _baseUrl;
	}

	@Override
	public void setBaseUrl( URL aBaseUrl ) {
		if ( aBaseUrl != null ) {
			if ( !aBaseUrl.getProtocol().equalsIgnoreCase( Protocol.HTTP.getName() ) && !aBaseUrl.getProtocol().equalsIgnoreCase( Protocol.HTTPS.getName() ) ) {
				throw new IllegalArgumentException( "Cannot use the protocol <" + aBaseUrl.getProtocol() + "> to do HTTP requests (" + aBaseUrl.toExternalForm() + "). You must provide a base URL for protocols <" + Protocol.HTTP.getName() + "> or <" + Protocol.HTTPS.getName() + ">." );
			}
			if ( aBaseUrl.getQuery() != null && aBaseUrl.getQuery().length() != 0 ) {
				throw new IllegalArgumentException( "Cannot use a query <" + aBaseUrl.getQuery() + "> as bayse path. You must provide a base URL without a query." );
			}
		}
		_baseUrl = aBaseUrl;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Pipe.
	 *
	 * @param aInputStream the a input stream
	 * @param aOutoutStream the a outout stream
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	protected static void pipe( InputStream aInputStream, OutputStream aOutoutStream ) throws IOException {
		int eBytes;
		byte[] theBuffer = new byte[PIPE_STREAM_BUFFER];
		while ( (eBytes = aInputStream.read( theBuffer )) > -1 ) {
			aOutoutStream.write( theBuffer, 0, eBytes );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class HttpRestRequestHandler.
	 */
	private class HttpRestRequestHandler implements RestRequestHandler {

		/**
		 * On rest request.
		 *
		 * @param aClientRequest the a client request
		 * @return the rest response
		 * @throws HttpResponseException the http response exception
		 */
		@Override
		public RestResponse onRestRequest( HttpClientRequest aClientRequest ) throws HttpResponseException {
			if ( _connectionStatus == ConnectionStatus.CLOSED ) {
				throw new IllegalStateException( " Expected a connection status <" + ConnectionStatus.OPENED.name() + ">, unable to produce a HTTP request while the connection is in status <" + _connectionStatus.name() + ">, did you forget to call #open(...)?" );
			}
			try {
				// |--> Create arguments for listener invocation:
				URL theUrl = null;
				String theQueryString = aClientRequest.getQueryFields().toHttpQueryString();
				if ( theQueryString == null ) theQueryString = "";
				try {
					theUrl = new URL( aClientRequest.getLocator() + theQueryString );
				}
				catch ( MalformedURLException e ) {

					if ( _baseUrl == null ) {
						throw new HttpResponseException( HttpStatusCode.INTERNAL_CLIENT_ERROR, "Unable to create a valid URL from locator <" + aClientRequest.getLocator() + ">: " + ExceptionUtility.toMessage( e ), e );
					}
					else {
						try {
							String theExternalForm = _baseUrl.toExternalForm();

							String theLocator = aClientRequest.getLocator();
							String theSeparator = "";
							if ( (theExternalForm == null || !theExternalForm.endsWith( "" + Delimiter.PATH.getChar() )) && (theLocator == null || !theLocator.startsWith( "" + Delimiter.PATH.getChar() )) ) {
								theSeparator = "" + Delimiter.PATH.getChar();
							}
							theUrl = new URL( theExternalForm + theSeparator + theLocator + theQueryString );
						}
						catch ( MalformedURLException e2 ) {
							throw new HttpResponseException( HttpStatusCode.INTERNAL_CLIENT_ERROR, "Unable to create a valid URL from the base URL <" + _baseUrl.toExternalForm() + "> and the locator <" + aClientRequest.getLocator() + ">: " + ExceptionUtility.toMessage( e2 ), e2 );
						}
					}
				}

				int thePort = theUrl.getPort();

				if ( thePort == -1 ) {
					String theProtocol = theUrl.getProtocol();
					if ( Protocol.HTTP.getName().equalsIgnoreCase( theProtocol ) ) {
						thePort = Port.HTTP.getValue();
					}
					if ( Protocol.HTTPS.getName().equalsIgnoreCase( theProtocol ) ) {
						thePort = Port.HTTPS.getValue();
					}
				}

				InetSocketAddress theRemoteAddress = new InetSocketAddress( theUrl.getHost(), thePort );
				// ---------------------------------------------------------
				// The port definitely is wrong as outgoing HTTP requests
				// may use any port available, though usually not the port
				// being targeted at! Unfortunately there is no way to
				// determine the originating InetSocketAddress from an
				// HttpURLConnection |-->
				InetSocketAddress theLocalAddress = InetSocketAddress.createUnresolved( InetAddress.getLocalHost().getHostName(), thePort );
				// <--| Unfortunately there is no way to determine the
				// originating InetSocketAddress from an HttpURLConnection
				// ---------------------------------------------------------
				// Create arguments for listener invocation <--|

				// |--> Do the HTTP request:
				HttpURLConnection theHttpConnection = (HttpURLConnection) theUrl.openConnection();

				theHttpConnection.setRequestMethod( aClientRequest.getHttpMethod().name() );
				List<ContentType> theAcceptTypes = aClientRequest.getHeaderFields().getAcceptTypes();
				if ( theAcceptTypes == null || theAcceptTypes.size() == 0 ) {
					aClientRequest.getHeaderFields().putAcceptTypes( getFactoryMediaTypes() );
				}

				theHttpConnection.setRequestMethod( aClientRequest.getHttpMethod().name() );

				// -------------------------------------------------------------
				// Correlation-ID:
				doRequestCorrelation( aClientRequest.getHeaderFields() );
				// -------------------------------------------------------------
				doSessionCorrelation( aClientRequest.getHeaderFields() );
				// -------------------------------------------------------------

				// -------------------------------------------------------------
				// As "HttpURLConnection#getRequestProperties()" is
				// immutable, we cannot go for this one:
				// -------------------------------------------------------------
				// theRestCaller.getHeaderFields().toHeaderFields(
				// theHttpConnection.getRequestProperties() );
				// -------------------------------------------------------------
				for ( String eKey : aClientRequest.getHeaderFields().keySet() ) {
					theHttpConnection.setRequestProperty( eKey, aClientRequest.getHeaderFields().toField( eKey ) );
				}

				// |--> Process the request:
				Object theRequest = aClientRequest.getRequest();
				// |--> Input-Stream:
				if ( (theRequest instanceof InputStream) ) {
					InputStream theInputStream = (InputStream) theRequest;
					theHttpConnection.setDoOutput( true );
					pipe( theInputStream, theHttpConnection.getOutputStream() );
					theHttpConnection.getOutputStream().flush();
				}
				// Input-Stream <--|
				// Object |-->
				else {
					String theBody = aClientRequest.toHttpBody();
					if ( theBody != null ) {
						theHttpConnection.setDoOutput( true );
						theHttpConnection.getOutputStream().write( theBody.getBytes() );
						theHttpConnection.getOutputStream().flush();
					}
				}

				if ( !aClientRequest.getHttpMethod().name().equalsIgnoreCase( theHttpConnection.getRequestMethod() ) ) {
					LOGGER.warn( "You issued a request with HTTP-Method <" + aClientRequest.getHttpMethod().name() + "> which is not applicable for sending a HTTP body, the HTTP-Method has been changed to <" + theHttpConnection.getRequestMethod() + ">." );
				}
				// Object <--|
				// Process the request <--|

				// |--> Process the response:
				ResponseHeaderFields theResponseHeaderFields = new ResponseHeaderFieldsImpl();
				Map<String, List<String>> theHeaderFields = theHttpConnection.getHeaderFields();
				theResponseHeaderFields.putAll( theHeaderFields );

				// -------------------------------------------------------------
				// Correlation-ID:
				doRequestCorrelation( theResponseHeaderFields );
				// -------------------------------------------------------------
				doSessionCorrelation( theResponseHeaderFields );
				// -------------------------------------------------------------

				HttpStatusCode theStatusCode = HttpStatusCode.toHttpStatusCode( theHttpConnection.getResponseCode() );
				// Process the response <--|
				// Do the HTTP request <--|

				// |--> Invoke the caller:
				RestResponse theResponse = new RestResponseImpl( theLocalAddress, theRemoteAddress, theStatusCode, theResponseHeaderFields, new HttpConnectionInputStream( theHttpConnection ), HttpRestClientImpl.this );
				// Invoke the caller <--|
				return theResponse;
			}
			catch ( IOException e ) {
				throw new HttpResponseException( HttpStatusCode.INTERNAL_CLIENT_ERROR, e, e.getMessage() );
			}
			catch ( BadRequestException e ) {
				throw new HttpResponseException( e.getStatusCode(), e, e.getMessage() );
			}
		}
	}

	/**
	 * Encapsulates the {@link HttpURLConnection} for retrieving the actual HTTP
	 * {@link InputStream} ({@link HttpURLConnection#getInputStream()}) or in
	 * case of an exception doing so encapsulates the exception's message in an
	 * {@link InputStream}.
	 *
	 */
	private static class HttpConnectionInputStream extends InputStream {

		private InputStream _inputStream;

		/**
		 * Instantiates a new http connection input stream.
		 *
		 * @param aHttpURLConnection the a http URL connection
		 */
		public HttpConnectionInputStream( HttpURLConnection aHttpURLConnection ) {
			try {
				_inputStream = aHttpURLConnection.getInputStream();
			}
			catch ( IOException e ) {
				String theMessage = e.getMessage();
				_inputStream = new ByteArrayInputStream( theMessage.getBytes( StandardCharsets.UTF_8 ) );
			}
		}

		/**
		 * Read.
		 *
		 * @return the int
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public int read() throws IOException {
			return _inputStream.read();
		}

		/**
		 * Hash code.
		 *
		 * @return the int
		 */
		@Override
		public int hashCode() {
			return _inputStream.hashCode();
		}

		/**
		 * Read.
		 *
		 * @param b the b
		 * @return the int
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public int read( byte[] b ) throws IOException {
			return _inputStream.read( b );
		}

		/**
		 * Equals.
		 *
		 * @param obj the obj
		 * @return true, if successful
		 */
		@Override
		public boolean equals( Object obj ) {
			return _inputStream.equals( obj );
		}

		/**
		 * Read.
		 *
		 * @param b the b
		 * @param off the off
		 * @param len the len
		 * @return the int
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public int read( byte[] b, int off, int len ) throws IOException {
			return _inputStream.read( b, off, len );
		}

		/**
		 * Skip.
		 *
		 * @param n the n
		 * @return the long
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public long skip( long n ) throws IOException {
			return _inputStream.skip( n );
		}

		/**
		 * To string.
		 *
		 * @return the string
		 */
		@Override
		public String toString() {
			return _inputStream.toString();
		}

		/**
		 * Available.
		 *
		 * @return the int
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public int available() throws IOException {
			return _inputStream.available();
		}

		/**
		 * Close.
		 *
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public void close() throws IOException {
			_inputStream.close();
		}

		/**
		 * Mark.
		 *
		 * @param readlimit the readlimit
		 */
		@Override
		public void mark( int readlimit ) {
			_inputStream.mark( readlimit );
		}

		/**
		 * Reset.
		 *
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public void reset() throws IOException {
			_inputStream.reset();
		}

		/**
		 * Mark supported.
		 *
		 * @return true, if successful
		 */
		@Override
		public boolean markSupported() {
			return _inputStream.markSupported();
		}
	}
}
