// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.FormFields;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.MediaTypeFactoryLookup;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.impls.HttpClientRequestImpl;
import org.refcodes.rest.RestCallerBuilder;
import org.refcodes.rest.RestClient;
import org.refcodes.rest.RestResponseEvent;
import org.refcodes.rest.RestResponseObserver;
import org.refcodes.textual.impls.VerboseTextBuilderImpl;

/**
 * The implementation of the {@link RestCallerBuilder} interface as good old
 * POJO for use by different {@link RestClient} implementations.
 */
public class RestCallerBuilderImpl extends HttpClientRequestImpl implements RestCallerBuilder {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private RestResponseObserver _responseObserver = null;

	private ConnectionStatus _connectionStatus = ConnectionStatus.NONE;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new rest caller builder impl.
	 *
	 * @param aHttpMethod the a http method
	 * @param aLocator the a locator
	 * @param aQueryFields the a query fields
	 * @param aHeaderFields the a Header-Fields
	 * @param aRequest the a request
	 * @param aResponseObserver the a response observer
	 * @param aMediaTypeFactoryLookup the a media type factory lookup
	 */
	public RestCallerBuilderImpl( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver, MediaTypeFactoryLookup aMediaTypeFactoryLookup ) {
		super( aHttpMethod, aLocator, aQueryFields, aHeaderFields, aRequest, aMediaTypeFactoryLookup );
		_responseObserver = aResponseObserver;
	}

	/**
	 * Instantiates a new rest caller builder impl.
	 *
	 * @param aHttpMethod the a http method
	 * @param aLocator the a locator
	 * @param aQueryFields the a query fields
	 * @param aResponseObserver the a response observer
	 * @param aMediaTypeFactoryLookup the a media type factory lookup
	 */
	public RestCallerBuilderImpl( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver, MediaTypeFactoryLookup aMediaTypeFactoryLookup ) {
		this( aHttpMethod, aLocator, aQueryFields, null, null, aResponseObserver, aMediaTypeFactoryLookup );
	}

	/**
	 * Instantiates a new rest caller builder impl.
	 *
	 * @param aHttpMethod the a http method
	 * @param aLocator the a locator
	 * @param aRequest the a request
	 * @param aResponseObserver the a response observer
	 * @param aMediaTypeFactoryLookup the a media type factory lookup
	 */
	public RestCallerBuilderImpl( HttpMethod aHttpMethod, String aLocator, Object aRequest, RestResponseObserver aResponseObserver, MediaTypeFactoryLookup aMediaTypeFactoryLookup ) {
		this( aHttpMethod, aLocator, null, null, aRequest, aResponseObserver, aMediaTypeFactoryLookup );
	}

	/**
	 * Instantiates a new rest caller builder impl.
	 *
	 * @param aHttpMethod the a http method
	 * @param aLocator the a locator
	 * @param aQueryFields the a query fields
	 * @param aRequest the a request
	 * @param aResponseObserver the a response observer
	 * @param aMediaTypeFactoryLookup the a media type factory lookup
	 */
	public RestCallerBuilderImpl( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver, MediaTypeFactoryLookup aMediaTypeFactoryLookup ) {
		this( aHttpMethod, aLocator, aQueryFields, null, aRequest, aResponseObserver, aMediaTypeFactoryLookup );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void onResponse( RestResponseEvent aResponse ) throws HttpResponseException {
		if ( _connectionStatus != ConnectionStatus.CLOSED ) {
			_responseObserver.onResponse( aResponse );
		}
		else {
			LOGGER.warn( "Ignoring response <" + aResponse + "> as this rest endpoint is in status <" + _connectionStatus + ">, you may have closed it already?" );
		}
	}

	@Override
	public String getLocator() {
		return _locator;
	}

	@Override
	public HttpMethod getHttpMethod() {
		return _httpMethod;
	}

	@Override
	public void setLocator( String aLocator ) {
		_locator = aLocator;
	}

	@Override
	public void setHttpMethod( HttpMethod aHttpMethod ) {
		_httpMethod = aHttpMethod;
	}

	@Override
	public RestResponseObserver getResponseObserver() {
		return _responseObserver;
	}

	@Override
	public void setResponseObserver( RestResponseObserver aLambda ) {
		_responseObserver = aLambda;
	}

	@Override
	public void open() throws OpenException {
		_connectionStatus = ConnectionStatus.OPENED;
		synchronized ( this ) {
			notifyAll();
		}
	}

	@Override
	public void close() throws CloseException {
		_connectionStatus = ConnectionStatus.CLOSED;
		synchronized ( this ) {
			notifyAll();
		}
	}

	@Override
	public ConnectionStatus getConnectionStatus() {
		return _connectionStatus;
	}

	@Override
	public void setQueryFields( FormFields aQueryFields ) {
		_queryFields = aQueryFields;
	}

	@Override
	public String toString() {
		return getClass().getSimpleName() + "(" + _httpMethod + ": " + _locator + "?" + new VerboseTextBuilderImpl().withElements( _queryFields ).toString() + ")@" + hashCode();
	}

	@Override
	public void setHeaderFields( RequestHeaderFields aHeaderFields ) {
		_headerFields = aHeaderFields;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
