package org.refcodes.rest.impls;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpServerResponse;
import org.refcodes.net.HttpStatusException;
import org.refcodes.rest.RestEndpointBuilder;
import org.refcodes.rest.RestRequestEvent;
import org.refcodes.rest.RestRequestObserver;
import org.refcodes.rest.RestServer;

/**
 * The implementation of the {@link RestEndpointBuilder} interface as good old
 * POJO for use by different {@link RestServer} implementations.
 */
public class RestEndpointBuilderImpl implements RestEndpointBuilder {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected RestRequestObserver _requestObserver = null;

	protected HttpMethod _httpMethod = HttpMethod.GET;

	protected String _locatorPattern = null;

	protected ConnectionStatus _connectionStatus = ConnectionStatus.NONE;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a plain {@link RestEndpointBuilder}, make sure to provide the
	 * least required attributes as demonstrated by the constructor
	 * {@link #RestEndpointBuilderImpl(HttpMethod, String, RestRequestObserver)}.
	 */
	public RestEndpointBuilderImpl() {}

	/**
	 * Constructs an {@link RestEndpointBuilder} with the least required
	 * attributes.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this
	 *        {@link RestEndpointBuilder} is bound.
	 * @param aLocatorPattern The local host's locator patter to which this
	 *        {@link RestEndpointBuilder} is bound. See
	 *        {@link #setLocatorPattern(String)} on the syntax of the pattern.
	 * @param aRequestObserver The listener processing requests targeted at this
	 *        {@link RestEndpointBuilder}.
	 */
	public RestEndpointBuilderImpl( HttpMethod aHttpMethod, String aLocatorPattern, RestRequestObserver aRequestObserver ) {
		_httpMethod = aHttpMethod;
		_locatorPattern = aLocatorPattern;
		_requestObserver = aRequestObserver;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	@Override
	public void setLocatorPattern( String aLocatorPattern ) {
		_locatorPattern = aLocatorPattern;
	}

	@Override
	public void setHttpMethod( HttpMethod aHttpMethod ) {
		_httpMethod = aHttpMethod;
	}

	@Override
	public void setRequestObserver( RestRequestObserver aLambda ) {
		_requestObserver = aLambda;
	}

	@Override
	public RestRequestObserver getRequestObserver() {
		return _requestObserver;
	}

	@Override
	public void onRequest( RestRequestEvent aRequest, HttpServerResponse aResponse ) throws HttpStatusException {
		if ( _connectionStatus == ConnectionStatus.OPENED ) {
			_requestObserver.onRequest( aRequest, aResponse );
		}
		else {
			LOGGER.warn( "Ignoring request <" + aRequest + "> as this rest endpoint is in status <" + _connectionStatus + ">, you may not have opened it?" );
		}
	}

	@Override
	public HttpMethod getHttpMethod() {
		return _httpMethod;
	}

	@Override
	public String getLocatorPattern() {
		return _locatorPattern;
	}

	@Override
	public void open() throws OpenException {
		_connectionStatus = ConnectionStatus.OPENED;
	}

	@Override
	public void close() throws CloseException {
		_connectionStatus = ConnectionStatus.CLOSED;
	}

	@Override
	public ConnectionStatus getConnectionStatus() {
		return _connectionStatus;
	}
}
