// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// =============================================================================
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// =============================================================================
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// together with the GPL linking exception applied; as being applied by the GNU
// Classpath ("http://www.gnu.org/software/classpath/license.html")
// =============================================================================
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// =============================================================================
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest.impls;

import static org.junit.Assert.*;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import org.junit.Ignore;
import org.junit.Test;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.impls.RuntimeLoggerFactorySingleton;
import org.refcodes.net.FormFields;
import org.refcodes.net.HttpBodyMap;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.MediaType;
import org.refcodes.net.ResponseCookie;
import org.refcodes.net.impls.FormFieldsImpl;
import org.refcodes.net.impls.PortManagerSingleton;
import org.refcodes.rest.HttpRestClient;
import org.refcodes.rest.HttpRestServer;
import org.refcodes.rest.RestCallerBuilder;
import org.refcodes.rest.RestResponse;
import org.refcodes.textual.HorizAlignTextMode;
import org.refcodes.textual.impls.HorizAlignTextBuilderImpl;

/**
 * The Class HttpRestClientTest.
 */
public class HttpRestClientTest {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int MOD_BYTE = 256;

	private static final String LOCATOR = "/bla";

	private static final String BASE_URL = "http://localhost";

	private static final String BASE_LOCATOR = "/refcodes";

	private static final String COOKIE_A_PATH = "/refcodes/bla";

	private static final String COOKIE_A_NAME = "refcodes";

	private static final String COOKIE_A_VALUE = "org";

	private static final String COOKIE_A_VALUE_2 = "com";

	private static final String COOKIE_B_PATH = "/refcodes/blub";

	private static final String COOKIE_B_NAME = "funcodes";

	private static final String COOKIE_B_VALUE = "forever";

	private static final String LAST_NAME = "Bushnell";

	private static final String FIRST_NAME = "Nolan";

	private static final String KEY_LAST_NAME = "lastName";

	private static final String KEY_FIRST_NAME = "firstName";

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Test asynchtronous rest client.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testAsynchtronousRestClient() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			LOGGER.debug( thePerson.toString() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test asynchtronous rest client 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testAsynchtronousRestClient2() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, new Person( FIRST_NAME, LAST_NAME ), ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			LOGGER.debug( thePerson.toString() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test synchtronous rest client.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws HttpResponseException the http response exception
	 */
	@Test
	public void testSynchtronousRestClient() throws IOException, HttpResponseException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestResponse aResponse = theRestClient.doRequest( HttpMethod.POST, LOCATOR, new Person( FIRST_NAME, LAST_NAME ) );
		Person thePerson = aResponse.getResponse( Person.class );
		LOGGER.debug( thePerson.toString() );
		theRestServer.closeQuietly();
		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test response builder over client.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws HttpResponseException the http response exception
	 */
	@Test
	public void testResponseBuilderOverClient() throws IOException, HttpResponseException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestResponse theResponse = theRestClient.buildRequest( HttpMethod.POST, LOCATOR, new Person( FIRST_NAME, LAST_NAME ) ).toRestResponse();
		Person thePerson = theResponse.getResponse( Person.class );
		LOGGER.debug( thePerson.toString() );
		theRestServer.closeQuietly();
		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test client cookie.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testClientCookie() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
			aResponse.getHeaderFields().addCookie( COOKIE_A_NAME, COOKIE_A_VALUE ).withPath( COOKIE_A_PATH );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			ResponseCookie theCookie = aResponse.getHeaderFields().getFirstCookie( COOKIE_A_NAME );
			assertEquals( COOKIE_A_NAME, theCookie.getKey() );
			assertEquals( COOKIE_A_VALUE, theCookie.getValue() );
			assertEquals( COOKIE_A_PATH, theCookie.getPath() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test client cookies.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testClientCookies() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
			aResponse.getHeaderFields().addCookie( COOKIE_A_NAME, COOKIE_A_VALUE ).withPath( COOKIE_A_PATH );
			aResponse.getHeaderFields().addCookie( COOKIE_A_NAME, COOKIE_A_VALUE_2 ).withPath( COOKIE_A_PATH );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			List<ResponseCookie> theCookies = aResponse.getHeaderFields().getCookies( COOKIE_A_NAME );
			assertEquals( "Expecting two cookies", 2, theCookies.size() );

			int a = 0;
			int b = 1;
			if ( theCookies.get( a ).getValue().equals( COOKIE_A_VALUE_2 ) ) {
				a = 1;
				b = 0;
			}

			assertEquals( COOKIE_A_NAME, theCookies.get( a ).getKey() );
			assertEquals( COOKIE_A_VALUE, theCookies.get( a ).getValue() );
			assertEquals( COOKIE_A_PATH, theCookies.get( a ).getPath() );
			assertEquals( COOKIE_A_NAME, theCookies.get( b ).getKey() );
			assertEquals( COOKIE_A_VALUE_2, theCookies.get( b ).getValue() );
			assertEquals( COOKIE_A_PATH, theCookies.get( b ).getPath() );
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test all client cookies.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testAllClientCookies() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			Person thePerson = aRequest.getRequest( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );
			aResponse.setResponse( thePerson );
			aResponse.getHeaderFields().addCookie( COOKIE_A_NAME, COOKIE_A_VALUE ).withPath( COOKIE_A_PATH );
			aResponse.getHeaderFields().addCookie( COOKIE_B_NAME, COOKIE_B_VALUE ).withPath( COOKIE_B_PATH );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			Person thePerson = aResponse.getResponse( Person.class );
			assertEquals( FIRST_NAME, thePerson.getFirstName() );
			assertEquals( LAST_NAME, thePerson.getLastName() );

			List<ResponseCookie> theCookies = aResponse.getHeaderFields().getAllCookies();
			assertEquals( "Expecting two cookies", 2, theCookies.size() );

			int a = 0;
			int b = 1;
			if ( theCookies.get( a ).getKey().equals( COOKIE_B_NAME ) ) {
				a = 1;
				b = 0;
			}

			assertEquals( COOKIE_A_NAME, theCookies.get( a ).getKey() );
			assertEquals( COOKIE_A_VALUE, theCookies.get( a ).getValue() );
			assertEquals( COOKIE_A_PATH, theCookies.get( a ).getPath() );
			assertEquals( COOKIE_B_NAME, theCookies.get( b ).getKey() );
			assertEquals( COOKIE_B_VALUE, theCookies.get( b ).getValue() );
			assertEquals( COOKIE_B_PATH, theCookies.get( b ).getPath() );

			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new Person( FIRST_NAME, LAST_NAME ) ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test web form.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testWebForm() throws IOException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			FormFields theFormFields = aRequest.getRequest( FormFields.class );
			assertEquals( FIRST_NAME, theFormFields.getFirst( KEY_FIRST_NAME ) );
			assertEquals( LAST_NAME, theFormFields.getFirst( KEY_LAST_NAME ) );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		FormFields theWebForm = new FormFieldsImpl().withAddTo( KEY_FIRST_NAME, FIRST_NAME ).withAddTo( KEY_LAST_NAME, LAST_NAME );
		RestCallerBuilder theCaller = theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( theWebForm );
		theCaller.getHeaderFields().putContentType( MediaType.APPLICATION_X_WWW_FORM_URLENCODED );
		theCaller.open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Digest the stream in the server's request lambda:.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testHttpRestClientStreamA() throws IOException {
		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );
		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			InputStream theInputStream = aRequest.getRequest( InputStream.class );
			try {
				int i = 0;
				int theCounter = 0;
				int eByte;
				StringBuffer theBuffer = new StringBuffer();
				while ( theInputStream.available() > 0 ) {
					eByte = theInputStream.read();
					assertEquals( theCounter % MOD_BYTE, eByte );
					theCounter++;
					theBuffer.append( new HorizAlignTextBuilderImpl().withColumnWidth( 3 ).withFillChar( '0' ).withHorizAlignTextMode( HorizAlignTextMode.RIGHT ).toString( eByte + "" ) + " " );
					i++;
					if ( i > 10 || theInputStream.available() == 0 ) {
						LOGGER.info( theBuffer.toString() );
						i = 0;
						theBuffer = new StringBuffer();
					}
				}
			}
			catch ( Exception e ) {
				fail();
			}

		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new NumberSequenceInputStream() ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Pass the stream back to the client and digest it the client's response
	 * lambda:.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	@Test
	public void testHttpRestClientStreamB() throws IOException {
		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );
		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			aResponse.setResponse( aRequest.getRequest( InputStream.class ) );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		theRestClient.doRequest( HttpMethod.POST, LOCATOR, ( aResponse ) -> {
			InputStream theInputStream = aResponse.getResponse( InputStream.class );
			try {
				int i = 0;
				int theCounter = 0;
				int eByte;
				StringBuffer theBuffer = new StringBuffer();
				while ( theInputStream.available() > 0 ) {
					eByte = theInputStream.read();
					assertEquals( theCounter % MOD_BYTE, eByte );
					theCounter++;
					theBuffer.append( new HorizAlignTextBuilderImpl().withColumnWidth( 3 ).withFillChar( '0' ).withHorizAlignTextMode( HorizAlignTextMode.RIGHT ).toString( eByte + "" ) + " " );
					i++;
					if ( i > 10 || theInputStream.available() == 0 ) {
						LOGGER.info( theBuffer.toString() );
						i = 0;
						theBuffer = new StringBuffer();
					}
				}
			}
			catch ( Exception e ) {
				fail();
			}

			theRestServer.closeQuietly();
			synchronized ( this ) {
				notifyAll();
			}
		} ).withRequest( new NumberSequenceInputStream() ).open();

		synchronized ( this ) {
			try {
				wait();
			}
			catch ( InterruptedException ignore ) {}
		}

		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test http body map 1.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws HttpResponseException the http response exception
	 */
	@Test
	public void testHttpBodyMap1() throws IOException, HttpResponseException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			HttpBodyMap thePerson = aRequest.getRequest( HttpBodyMap.class );
			assertEquals( FIRST_NAME, thePerson.get( "/firstName" ) );
			assertEquals( LAST_NAME, thePerson.get( "/lastName" ) );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestResponse aResponse = theRestClient.doRequest( HttpMethod.POST, LOCATOR, new Person( FIRST_NAME, LAST_NAME ) );
		Person thePerson = aResponse.getResponse( Person.class );
		LOGGER.debug( thePerson.toString() );
		theRestServer.closeQuietly();
		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test http body map 2.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws HttpResponseException the http response exception
	 */
	@Test
	public void testHttpBodyMap2() throws IOException, HttpResponseException {

		Integer thePort = PortManagerSingleton.getInstance().bindAnyPort();
		LOGGER.info( "Using port <" + thePort + "> for testing ..." );

		HttpRestServer theRestServer = new HttpRestServerImpl().withBaseLocator( BASE_LOCATOR );
		theRestServer.onPost( LOCATOR, ( aRequest, aResponse ) -> {
			HttpBodyMap thePerson = aRequest.getRequest( HttpBodyMap.class );
			assertEquals( FIRST_NAME, thePerson.get( "/firstName" ) );
			assertEquals( LAST_NAME, thePerson.get( "/lastName" ) );
			aResponse.setResponse( thePerson );
		} ).open();
		theRestServer.open( thePort );

		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( toBaseUrl( thePort ) );
		RestResponse aResponse = theRestClient.doRequest( HttpMethod.POST, LOCATOR, new Person( FIRST_NAME, LAST_NAME ) );
		HttpBodyMap thePerson = aResponse.getResponse( HttpBodyMap.class );
		LOGGER.debug( thePerson.toString() );
		assertEquals( FIRST_NAME, thePerson.get( "/firstName" ) );
		assertEquals( LAST_NAME, thePerson.get( "/lastName" ) );
		theRestServer.closeQuietly();
		PortManagerSingleton.getInstance().unbindPort( thePort );
	}

	/**
	 * Test edge case.
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 * @throws HttpResponseException the http response exception
	 */
	@Ignore
	@Test
	public void testEdgeCase() throws IOException, HttpResponseException {
		HttpRestClient theRestClient = new HttpRestClientImpl().withBaseUrl( "https://www.heise.de" );
		RestResponse theResponse = theRestClient.doGet( "thema/Linux-und-Open-Source" );
		System.out.println( theResponse.getHttpBody() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * To base url.
	 *
	 * @param thePort the the port
	 * @return the string
	 */
	private String toBaseUrl( Integer thePort ) {
		return BASE_URL + ":" + thePort + BASE_LOCATOR;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * The Class Person.
	 */
	public static class Person {

		/**
		 * To string.
		 *
		 * @return the string
		 */
		@Override
		public String toString() {
			return "Person [firstName=" + firstName + ", lastName=" + lastName + "]";
		}

		private String firstName;
		private String lastName;

		/**
		 * Instantiates a new person.
		 */
		public Person() {};

		/**
		 * Instantiates a new person.
		 *
		 * @param firstName the first name
		 * @param lastName the last name
		 */
		public Person( String firstName, String lastName ) {
			this.firstName = firstName;
			this.lastName = lastName;
		}

		/**
		 * Gets the first name.
		 *
		 * @return the first name
		 */
		public String getFirstName() {
			return firstName;
		}

		/**
		 * Sets the first name.
		 *
		 * @param firstName the new first name
		 */
		public void setFirstName( String firstName ) {
			this.firstName = firstName;
		}

		/**
		 * Gets the last name.
		 *
		 * @return the last name
		 */
		public String getLastName() {
			return lastName;
		}

		/**
		 * Sets the last name.
		 *
		 * @param lastName the new last name
		 */
		public void setLastName( String lastName ) {
			this.lastName = lastName;
		}
	}

	/**
	 * The Class NumberSequenceInputStream.
	 */
	private static class NumberSequenceInputStream extends InputStream {

		private int _counter = 0;

		/**
		 * Read.
		 *
		 * @return the int
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public int read() throws IOException {
			if ( available() <= 0 ) return -1;
			return _counter++ % MOD_BYTE;
		}

		/**
		 * Available.
		 *
		 * @return the int
		 * @throws IOException Signals that an I/O exception has occurred.
		 */
		@Override
		public int available() throws IOException {
			return 1024 - _counter;
		}

	}
}
