// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.security.KeyStore;

import org.refcodes.net.Url;
import org.refcodes.rest.HttpRegistryContext.HttpRegistryContextBuilder;
import org.refcodes.security.TrustStoreDescriptor;

/**
 *
 */
public abstract class AbstractHttpRegistryContextBuilder<DESC extends HttpServerDescriptor> implements HttpRegistryContextBuilder<DESC> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private Url _registryUrl;
	private TrustStoreDescriptor _trustStoreDescriptor;
	private DESC _serverDescriptor;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs the {@link AbstractHttpRegistryContextBuilder} context.
	 * 
	 * @param aRegistryUrl The {@link Url} pointing to the service discovery
	 *        registry.
	 * @param aServerDescriptor THe descriptor describing the server ("service")
	 *        to be discovered.
	 */
	public AbstractHttpRegistryContextBuilder( Url aRegistryUrl, DESC aServerDescriptor ) {
		this( aRegistryUrl, null, aServerDescriptor );
	}

	/**
	 * Constructs the {@link AbstractHttpRegistryContextBuilder} context.
	 * 
	 * @param aRegistryUrl The {@link Url} pointing to the service discovery
	 *        registry.
	 * @param aStoreDescriptor The descriptor describing the truststore
	 *        ({@link KeyStore}) required for establishing an HTTPS connection
	 *        to the registry.
	 * @param aServerDescriptor THe descriptor describing the server ("service")
	 *        to be discovered.
	 */
	public AbstractHttpRegistryContextBuilder( Url aRegistryUrl, TrustStoreDescriptor aStoreDescriptor, DESC aServerDescriptor ) {
		_registryUrl = aRegistryUrl;
		_serverDescriptor = aServerDescriptor;
		_trustStoreDescriptor = aStoreDescriptor;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Url getHttpRegistryUrl() {
		return _registryUrl;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DESC getHttpServerDescriptor() {
		return _serverDescriptor;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TrustStoreDescriptor getTrustStoreDescriptor() {
		return _trustStoreDescriptor;
	}

	@Override
	public void setHttpRegistryUrl( Url aRegistryUrl ) {
		_registryUrl = aRegistryUrl;

	}

	@Override
	public void setHttpServerDescriptor( DESC aServerDescriptor ) {
		_serverDescriptor = aServerDescriptor;

	}

	@Override
	public void setTrustStoreDescriptor( TrustStoreDescriptor aStoreDescriptor ) {
		_trustStoreDescriptor = aStoreDescriptor;
	}

	@Override
	public HttpRegistryContextBuilder<DESC> withHttpRegistryUrl( Url aUrl ) {
		setHttpRegistryUrl( aUrl );
		return this;
	}

	@Override
	public HttpRegistryContextBuilder<DESC> withHttpServerDescriptor( DESC aServerDescriptor ) {
		setHttpServerDescriptor( aServerDescriptor );
		return this;
	}

	@Override
	public HttpRegistryContextBuilder<DESC> withTrustStoreDescriptor( TrustStoreDescriptor aStoreDescriptor ) {
		setTrustStoreDescriptor( aStoreDescriptor );
		return this;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
