// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.net.MalformedURLException;
import java.net.URL;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.data.Scheme;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.MediaType;
import org.refcodes.net.MediaTypeFactory;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.Url;
import org.refcodes.security.TrustStoreDescriptor;

/**
 * Abstract class for easily decorating a {@link HttpRestClient}.
 */
public abstract class AbstractHttpRestClientDecorator<B extends HttpRestClient> implements HttpRestClient {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private HttpRestClient _client;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Decorates the given {@link HttpRestClient}.
	 * 
	 * @param aClient The {@link HttpRestClient} to be decorated.
	 */
	public AbstractHttpRestClientDecorator( HttpRestClient aClient ) {
		_client = aClient;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionStatus getConnectionStatus() {
		return _client.getConnectionStatus();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean addMediaTypeFactory( MediaTypeFactory aMediaTypeFactory ) {
		return _client.addMediaTypeFactory( aMediaTypeFactory );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		_client.close();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void closeIn( int aCloseInMillis ) {
		_client.closeIn( aCloseInMillis );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void closeQuietly() {
		_client.closeQuietly();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void closeUnchecked() {
		_client.closeUnchecked();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void disableRequestCorrelation() {
		_client.disableRequestCorrelation();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void disableSessionCorrelation() {
		_client.disableSessionCorrelation();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void enableRequestCorrelation() {
		_client.enableRequestCorrelation();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void enableSessionCorrelation() {
		_client.enableSessionCorrelation();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Url getBaseUrl() {
		return _client.getBaseUrl();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public MediaType[] getFactoryMediaTypes() {
		return _client.getFactoryMediaTypes();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TrustStoreDescriptor getTrustStoreDescriptor() {
		return _client.getTrustStoreDescriptor();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getUserAgent() {
		return _client.getUserAgent();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasMediaTypeFactory( MediaType aMediaType ) {
		return _client.hasMediaTypeFactory( aMediaType );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasRequestCorrelation() {
		return _client.hasRequestCorrelation();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasSessionCorrelation() {
		return _client.hasSessionCorrelation();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws OpenException {
		_client.open();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open( Url aBaseUrl, TrustStoreDescriptor aStoreDescriptor ) throws OpenException {
		_client.open( getBaseUrl(), aStoreDescriptor );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void openUnchecked() {
		_client.openUnchecked();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( Scheme aProtocol, String aHost ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( Scheme aProtocol, String aHost, int aPort ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPort );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( Scheme aProtocol, String aHost, int aPort, String aPath ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPort, aPath );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( Scheme aProtocol, String aHost, String aPath ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPath );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( String aBaseUrl ) throws MalformedURLException {
		_client.setBaseUrl( aBaseUrl );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( String aProtocol, String aHost ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( String aProtocol, String aHost, int aPort ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPort );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( String aProtocol, String aHost, int aPort, String aPath ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPort, aPath );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( String aProtocol, String aHost, String aPath ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPath );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( Url aBaseUrl ) {
		_client.setBaseUrl( aBaseUrl );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBaseUrl( URL aBaseUrl ) {
		_client.setBaseUrl( aBaseUrl );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setRequestCorrelation( boolean hasRequestCorrelation ) {
		_client.setRequestCorrelation( hasRequestCorrelation );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSessionCorrelation( boolean hasSessionCorrelation ) {
		_client.setSessionCorrelation( hasSessionCorrelation );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTrustStoreDescriptor( TrustStoreDescriptor aStoreDescriptor ) {
		_client.setTrustStoreDescriptor( aStoreDescriptor );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setUserAgent( String aUserAgent ) {
		_client.setUserAgent( aUserAgent );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public MediaTypeFactory toMediaTypeFactory( MediaType aMediaType ) {
		return _client.toMediaTypeFactory( aMediaType );
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( Scheme aProtocol, String aHost ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( Scheme aProtocol, String aHost, int aPort ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPort );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( Scheme aProtocol, String aHost, int aPort, String aPath ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPort, aPath );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( Scheme aProtocol, String aHost, String aPath ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPath );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( String aBaseUrl ) throws MalformedURLException {
		_client.setBaseUrl( aBaseUrl );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( String aProtocol, String aHost ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( String aProtocol, String aHost, int aPort ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPort );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( String aProtocol, String aHost, int aPort, String aPath ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPort, aPath );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( String aProtocol, String aHost, String aPath ) throws MalformedURLException {
		_client.setBaseUrl( aProtocol, aHost, aPath );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( Url aBaseUrl ) {
		_client.setBaseUrl( aBaseUrl );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withBaseUrl( URL aBaseURL ) {
		_client.setBaseUrl( aBaseURL );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withDisableRequestCorrelation() {
		_client.disableRequestCorrelation();
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withDisableSessionCorrelation() {
		_client.disableSessionCorrelation();
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withEnableRequestCorrelation() {
		_client.enableRequestCorrelation();
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withEnableSessionCorrelation() {
		_client.enableSessionCorrelation();
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withRequestCorrelation( boolean hasRequestCorrelation ) {
		_client.setRequestCorrelation( hasRequestCorrelation );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withSessionCorrelation( boolean hasSessionCorrelation ) {
		_client.setSessionCorrelation( hasSessionCorrelation );
		return (B) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("unchecked")
	@Override
	public B withUserAgent( String aUserAgent ) {
		_client.setUserAgent( aUserAgent );
		return (B) this;
	}

	@Override
	public RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return _client.buildRequest( aHttpMethod, toUrl( aUrl ), aHeaderFields, aRequest );
	}

	@Override
	public RestResponse doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return _client.doRequest( aHttpMethod, toUrl( aUrl ), aHeaderFields, aRequest );
	}

	@Override
	public RestCallerBuilder doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return _client.doRequest( aHttpMethod, toUrl( aUrl ), aHeaderFields, aRequest, aResponseObserver );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Hook for sub-classes to modify the request URL.
	 * 
	 * @param aUrl The {@link Url} for the request.
	 * 
	 * @return The tinkered {@link Url}, by default it returns the provided
	 *         {@link Url} unmodified.
	 */
	protected Url toUrl( Url aUrl ) {
		return aUrl;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
