package org.refcodes.rest;

import java.net.InetSocketAddress;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;
import org.refcodes.net.BasicAuthCredentials;
import org.refcodes.net.BasicAuthObserver;
import org.refcodes.net.BasicAuthResponse;
import org.refcodes.net.HttpMethod;

/**
 * The implementation of the {@link BasicAuthEndpointBuilder} interface as good
 * old POJO for use by different {@link RestServer} implementations.
 */
public class BasicAuthEndpointBuilderImpl implements BasicAuthEndpointBuilder {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	protected BasicAuthObserver _requestObserver = null;

	protected HttpMethod _httpMethod = HttpMethod.GET;

	protected String _locatorPattern = null;

	protected ConnectionStatus _connectionStatus = ConnectionStatus.NONE;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs a plain {@link BasicAuthEndpointBuilder}, make sure to provide
	 * the least required attributes as demonstrated by the constructor
	 * {@link #BasicAuthEndpointBuilderImpl(HttpMethod, String, BasicAuthObserver)}.
	 */
	public BasicAuthEndpointBuilderImpl() {}

	/**
	 * Constructs an {@link BasicAuthEndpointBuilder} with the least required
	 * attributes.
	 * 
	 * @param aHttpMethod The HTTP-Method to which this
	 *        {@link BasicAuthEndpointBuilder} is bound.
	 * @param aLocatorPattern The local host's locator patter to which this
	 *        {@link BasicAuthEndpointBuilder} is bound. See
	 *        {@link #setLocatorPattern(String)} on the syntax of the pattern.
	 * @param aRequestObserver The listener processing requests targeted at this
	 *        {@link BasicAuthEndpointBuilder}.
	 */
	public BasicAuthEndpointBuilderImpl( HttpMethod aHttpMethod, String aLocatorPattern, BasicAuthObserver aRequestObserver ) {
		_httpMethod = aHttpMethod;
		_locatorPattern = aLocatorPattern;
		_requestObserver = aRequestObserver;
	}

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setLocatorPattern( String aLocatorPattern ) {
		_locatorPattern = aLocatorPattern;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHttpMethod( HttpMethod aHttpMethod ) {
		_httpMethod = aHttpMethod;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setRequestObserver( BasicAuthObserver aLambda ) {
		_requestObserver = aLambda;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public BasicAuthObserver getRequestObserver() {
		return _requestObserver;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public BasicAuthResponse onBasicAuthRequest( InetSocketAddress aLocalAddress, InetSocketAddress aRemoteAddress, HttpMethod aHttpMethod, String aLocator, BasicAuthCredentials aCredentials, String aRealm ) {
		if ( _connectionStatus == ConnectionStatus.OPENED ) {
			return _requestObserver.onBasicAuthRequest( aLocalAddress, aRemoteAddress, aHttpMethod, aLocator, aCredentials, aRealm );
		}
		else {
			LOGGER.warn( "Ignoring Basic-Auth <" + aHttpMethod + "> request for locator <" + aLocator + "> as this rest endpoint is in status <" + _connectionStatus + ">, you may not have opened it?" );
		}
		return BasicAuthResponse.BASIC_AUTH_SKIP;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HttpMethod getHttpMethod() {
		return _httpMethod;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getLocatorPattern() {
		return _locatorPattern;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws OpenException {
		_connectionStatus = ConnectionStatus.OPENED;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		_connectionStatus = ConnectionStatus.CLOSED;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionStatus getConnectionStatus() {
		return _connectionStatus;
	}
}
