// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import org.refcodes.component.CloseException;
import org.refcodes.component.ConnectionStatus;
import org.refcodes.component.OpenException;
import org.refcodes.logger.RuntimeLogger;
import org.refcodes.logger.RuntimeLoggerFactorySingleton;
import org.refcodes.net.HttpClientRequestImpl;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.MediaTypeFactoryLookup;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.Url;
import org.refcodes.net.UrlBuilderImpl;
import org.refcodes.textual.VerboseTextBuilderImpl;

/**
 * The implementation of the {@link RestCallerBuilder} interface as good old
 * POJO for use by different {@link RestClient} implementations.
 */
public class RestCallerBuilderImpl extends HttpClientRequestImpl implements RestCallerBuilder {

	private static RuntimeLogger LOGGER = RuntimeLoggerFactorySingleton.createRuntimeLogger();

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private RestResponseObserver _responseObserver = null;

	private ConnectionStatus _connectionStatus = ConnectionStatus.NONE;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new rest caller builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aHeaderFields the Header-Fields
	 * @param aRequest the request
	 * @param aResponseObserver the response observer
	 * @param aMediaTypeFactoryLookup the media type factory lookup
	 */
	public RestCallerBuilderImpl( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver, MediaTypeFactoryLookup aMediaTypeFactoryLookup ) {
		super( aHttpMethod, aUrl instanceof UrlBuilder ? aUrl : new UrlBuilderImpl( aUrl ), aHeaderFields, aRequest, aMediaTypeFactoryLookup );
		_responseObserver = aResponseObserver;
	}

	/**
	 * Instantiates a new rest caller builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aResponseObserver the response observer
	 * @param aMediaTypeFactoryLookup the media type factory lookup
	 */
	public RestCallerBuilderImpl( HttpMethod aHttpMethod, Url aUrl, RestResponseObserver aResponseObserver, MediaTypeFactoryLookup aMediaTypeFactoryLookup ) {
		this( aHttpMethod, aUrl, null, null, aResponseObserver, aMediaTypeFactoryLookup );
	}

	/**
	 * Instantiates a new rest caller builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aRequest the request
	 * @param aResponseObserver the response observer
	 * @param aMediaTypeFactoryLookup the media type factory lookup
	 */
	public RestCallerBuilderImpl( HttpMethod aHttpMethod, Url aUrl, Object aRequest, RestResponseObserver aResponseObserver, MediaTypeFactoryLookup aMediaTypeFactoryLookup ) {
		this( aHttpMethod, aUrl, null, aRequest, aResponseObserver, aMediaTypeFactoryLookup );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onResponse( RestResponseEvent aResponse ) throws HttpResponseException {
		if ( _connectionStatus != ConnectionStatus.CLOSED ) {
			_responseObserver.onResponse( aResponse );
		}
		else {
			LOGGER.warn( "Ignoring response <" + aResponse + "> as this rest endpoint is in status <" + _connectionStatus + ">, you may have closed it already?" );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HttpMethod getHttpMethod() {
		return _httpMethod;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHttpMethod( HttpMethod aHttpMethod ) {
		_httpMethod = aHttpMethod;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RestResponseObserver getResponseObserver() {
		return _responseObserver;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setResponseObserver( RestResponseObserver aLambda ) {
		_responseObserver = aLambda;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws OpenException {
		_connectionStatus = ConnectionStatus.OPENED;
		synchronized ( this ) {
			notifyAll();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws CloseException {
		_connectionStatus = ConnectionStatus.CLOSED;
		synchronized ( this ) {
			notifyAll();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ConnectionStatus getConnectionStatus() {
		return _connectionStatus;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return getClass().getSimpleName() + "(" + _httpMethod + ": " + _url.toHttpUrl() + "?" + new VerboseTextBuilderImpl().withElements( _url.getQueryFields() ).toString() + ")@" + hashCode();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHeaderFields( RequestHeaderFields aHeaderFields ) {
		_headerFields = aHeaderFields;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Url getUrl() {
		return _url;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setUrl( Url aUrl ) {
		_url = aUrl;
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
