// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.net.MalformedURLException;
import java.util.Timer;
import java.util.TimerTask;

import org.refcodes.data.IoRetryCount;
import org.refcodes.data.IoTimeout;
import org.refcodes.net.BasicAuthCredentials;
import org.refcodes.net.BasicAuthCredentialsImpl;
import org.refcodes.net.FormFields;
import org.refcodes.net.FormFieldsImpl;
import org.refcodes.net.GrantType;
import org.refcodes.net.HttpBodyMap;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.HttpStatusException;
import org.refcodes.net.MediaType;
import org.refcodes.net.OauthField;
import org.refcodes.net.OauthToken;
import org.refcodes.net.OauthTokenImpl;
import org.refcodes.net.Url;

/**
 * Self refreshing implementation of the {@link OauthToken}. In case a refresh
 * token (as of {@link #getRefreshToken()} and has been provided, then the
 * access token (as of {@link #getAccessToken()} is refreshed within the
 * "expires in" time (as of {@link #getExpiresIn()}. The refresh daemon
 * terminates and this instance is disposed when the provided
 * {@link HttpRestClient}'s {@link HttpRestClient#close()} method is called or
 * the {@link #dispose()} method is invoked.
 */
public class OauthTokenHandler extends OauthTokenImpl implements OauthToken {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	private static final int DEFAULT_EXPIRES_IN = 60 * 3;
	private static final int MAX_REFRESH_TRIES = IoRetryCount.MIN.getNumber();
	private static final long MAX_RETRY_SLEEP_TIME = IoTimeout.MIN.getMilliseconds();

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private String _url;
	private HttpRestClient _restClient;
	private Timer _timer;
	private TimerTask _timerTask;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters by using the user's name and password for a "password" grant
	 * type authentication to retrieve a {@link OauthToken}.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aUserName The user's name for "password" grant type.
	 * @param aUserPassword The user's password for "password" grant type.
	 * @param aClientId The client's ID for "password" grant type.
	 * @param aClientSecret The client's secret for "password" grant type.
	 *
	 * @throws MalformedURLException thrown in case of a malformed URL.
	 * @throws HttpStatusException thrown in case a HTTP response was of an
	 *         erroneous status.
	 */
	public OauthTokenHandler( Url aUrl, String aClientId, String aClientSecret, String aUserName, String aUserPassword ) throws HttpStatusException, MalformedURLException {
		this( aUrl.toHttpUrl(), new HttpRestClientImpl(), aClientId, aClientSecret, aUserName, aUserPassword );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters by using the user's name and password for a "password" grant
	 * type authentication to retrieve a {@link OauthToken}.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aHttpRestClient The {@link HttpRestClient} to use when refreshing
	 *        the token.
	 * @param aUserName The user's name for "password" grant type.
	 * @param aUserPassword The user's password for "password" grant type.
	 * @param aClientId The client's ID for "password" grant type.
	 * @param aClientSecret The client's secret for "password" grant type.
	 * 
	 * @throws MalformedURLException thrown in case of a malformed URL.
	 * @throws HttpStatusException thrown in case a HTTP response was of an
	 *         erroneous status.
	 */
	public OauthTokenHandler( Url aUrl, HttpRestClient aHttpRestClient, String aClientId, String aClientSecret, String aUserName, String aUserPassword ) throws HttpStatusException, MalformedURLException {
		this( aUrl.toHttpUrl(), aHttpRestClient, aClientId, aClientSecret, aUserName, aUserPassword );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters by using the user's name and password for a "password" grant
	 * type authentication to retrieve a {@link OauthToken}.
	 * 
	 * @param aUrl The URL to be used when refreshing the access token.
	 * @param aUserName The user's name for "password" grant type.
	 * @param aUserPassword The user's password for "password" grant type.
	 * @param aClientId The client's ID for "password" grant type.
	 * @param aClientSecret The client's secret for "password" grant type.
	 * 
	 * @throws MalformedURLException thrown in case of a malformed URL.
	 * @throws HttpStatusException thrown in case a HTTP response was of an
	 *         erroneous status.
	 */
	public OauthTokenHandler( String aUrl, String aClientId, String aClientSecret, String aUserName, String aUserPassword ) throws HttpStatusException, MalformedURLException {
		this( aUrl, new HttpRestClientImpl(), aClientId, aClientSecret, aUserName, aUserPassword );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters by using the user's name and password for a "password" grant
	 * type authentication to retrieve a {@link OauthToken}.
	 * 
	 * @param aUrl The URL to be used when refreshing the access token.
	 * @param aHttpRestClient The {@link HttpRestClient} to use when refreshing
	 *        the token.
	 * @param aUserName The user's name for "password" grant type.
	 * @param aUserPassword The user's password for "password" grant type.
	 * @param aClientId The client's ID for "password" grant type.
	 * @param aClientSecret The client's secret for "password" grant type.
	 * 
	 * @throws MalformedURLException thrown in case of a malformed URL.
	 * @throws HttpStatusException thrown in case a HTTP response was of an
	 *         erroneous status.
	 */
	public OauthTokenHandler( String aUrl, HttpRestClient aHttpRestClient, String aClientId, String aClientSecret, String aUserName, String aUserPassword ) throws HttpStatusException, MalformedURLException {
		this( aUrl, aHttpRestClient, toOauthToken( aUrl, aHttpRestClient, aClientId, aClientSecret, aUserName, aUserPassword ) );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters. For the refresh token parameters, please refer to
	 * "https://www.oauth.com/oauth2-servers/access-tokens/refreshing-access-tokens".
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aAccessToken The access token to be stored (as of
	 *        {@link #getAccessToken()}).
	 * @param aRefreshToken The refresh token to be used (as of
	 *        {@link #getRefreshToken()}).
	 * @param aTokenType The token type to be stored (as of
	 *        {@link #getTokenType()}).
	 * @param aExpiresIn The "expires in" time to be stored (as of
	 *        {@link #getExpiresIn()}).
	 * @param aScope The scope to be stored (as of {@link #getScope()}).
	 */
	public OauthTokenHandler( String aUrl, String aAccessToken, String aRefreshToken, String aTokenType, Integer aExpiresIn, String aScope ) {
		super( aAccessToken, aRefreshToken, aTokenType, aExpiresIn, aScope );
		initialize( aUrl, new HttpRestClientImpl() );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aAccessToken The access token to be stored (as of
	 *        {@link #getAccessToken()}).
	 * @param aRefreshToken The refresh token to be used (as of
	 *        {@link #getRefreshToken()}).
	 * @param aTokenType The token type to be stored (as of
	 *        {@link #getTokenType()}).
	 * @param aExpiresIn The "expires in" time to be stored (as of
	 *        {@link #getExpiresIn()}).
	 * @param aScope The scope to be stored (as of {@link #getScope()}).
	 */
	public OauthTokenHandler( Url aUrl, String aAccessToken, String aRefreshToken, String aTokenType, Integer aExpiresIn, String aScope ) {
		super( aAccessToken, aRefreshToken, aTokenType, aExpiresIn, aScope );
		initialize( aUrl.toHttpUrl(), new HttpRestClientImpl() );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aOauthToken The {@link HttpBodyMap} containing the OAuth token
	 *        with the required information.
	 */
	public OauthTokenHandler( Url aUrl, HttpBodyMap aOauthToken ) {
		super( aOauthToken );
		initialize( aUrl.toHttpUrl(), new HttpRestClientImpl() );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aOauthToken The {@link OauthToken} containing the required
	 *        information.
	 */
	public OauthTokenHandler( Url aUrl, OauthToken aOauthToken ) {
		super( aOauthToken );
		initialize( aUrl.toHttpUrl(), new HttpRestClientImpl() );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters. For the refresh token parameters, please refer to
	 * "https://www.oauth.com/oauth2-servers/access-tokens/refreshing-access-tokens".
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aOauthToken The {@link OauthToken} containing the required
	 *        information.
	 */
	public OauthTokenHandler( String aUrl, OauthToken aOauthToken ) {
		super( aOauthToken );
		initialize( aUrl, new HttpRestClientImpl() );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aHttpRestClient The {@link HttpRestClient} to use when refreshing
	 *        the token.
	 * @param aAccessToken The access token to be stored (as of
	 *        {@link #getAccessToken()}).
	 * @param aRefreshToken The refresh token to be used (as of
	 *        {@link #getRefreshToken()}).
	 * @param aTokenType The token type to be stored (as of
	 *        {@link #getTokenType()}).
	 * @param aExpiresIn The "expires in" time to be stored (as of
	 *        {@link #getExpiresIn()}).
	 * @param aScope The scope to be stored (as of {@link #getScope()}).
	 */
	public OauthTokenHandler( Url aUrl, HttpRestClient aHttpRestClient, String aAccessToken, String aRefreshToken, String aTokenType, Integer aExpiresIn, String aScope ) {
		super( aAccessToken, aRefreshToken, aTokenType, aExpiresIn, aScope );
		initialize( aUrl.toHttpUrl(), aHttpRestClient );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aHttpRestClient The {@link HttpRestClient} to use when refreshing
	 *        the token.
	 * @param aOauthToken The {@link HttpBodyMap} containing the OAuth token
	 *        with the required information.
	 */
	public OauthTokenHandler( Url aUrl, HttpRestClient aHttpRestClient, HttpBodyMap aOauthToken ) {
		super( aOauthToken );
		initialize( aUrl.toHttpUrl(), aHttpRestClient );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters.
	 * 
	 * @param aUrl The URL to be used when refreshing the access token.
	 * @param aHttpRestClient The {@link HttpRestClient} to use when refreshing
	 *        the token.
	 * @param aOauthToken The {@link OauthToken} containing the required
	 *        information.
	 */
	public OauthTokenHandler( String aUrl, HttpRestClient aHttpRestClient, OauthToken aOauthToken ) {
		super( aOauthToken );
		initialize( aUrl, aHttpRestClient );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aHttpRestClient The {@link HttpRestClient} to use when refreshing
	 *        the token.
	 * @param aOauthToken The {@link OauthToken} containing the required
	 *        information.
	 */
	public OauthTokenHandler( Url aUrl, HttpRestClient aHttpRestClient, OauthToken aOauthToken ) {
		super( aOauthToken );
		initialize( aUrl.toHttpUrl(), aHttpRestClient );
	}

	/**
	 * Constructs an instance of the {@link OauthTokenHandler} with the given
	 * parameters. For the refresh token parameters, please refer to
	 * "https://www.oauth.com/oauth2-servers/access-tokens/refreshing-access-tokens".
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aHttpRestClient The {@link HttpRestClient} to use when refreshing
	 *        the token.
	 * @param aAccessToken The access token to be stored (as of
	 *        {@link #getAccessToken()}).
	 * @param aRefreshToken The refresh token to be used (as of
	 *        {@link #getRefreshToken()}).
	 * @param aTokenType The token type to be stored (as of
	 *        {@link #getTokenType()}).
	 * @param aExpiresIn The "expires in" time to be stored (as of
	 *        {@link #getExpiresIn()}).
	 * @param aScope The scope to be stored (as of {@link #getScope()}).
	 */
	public OauthTokenHandler( String aUrl, HttpRestClient aHttpRestClient, String aAccessToken, String aRefreshToken, String aTokenType, Integer aExpiresIn, String aScope ) {
		super( aAccessToken, aRefreshToken, aTokenType, aExpiresIn, aScope );
		initialize( aUrl, aHttpRestClient );
	}

	/**
	 * Initializer supporting the CTOR.
	 * 
	 * @param aUrl The {@link Url} to be used when refreshing the access token.
	 * @param aHttpRestClient The {@link HttpRestClient} to use when refreshing
	 *        the token.
	 */
	private void initialize( String aUrl, HttpRestClient aHttpRestClient ) {
		if ( _refreshToken != null ) {
			Integer theExpiresIn = _expiresIn;
			if ( theExpiresIn == null || theExpiresIn == -1 ) {
				theExpiresIn = DEFAULT_EXPIRES_IN;
			}
			_url = aUrl;
			_restClient = aHttpRestClient;
			_timer = new Timer( true );
			_timerTask = new RefreshTask();
			_timer.schedule( _timerTask, toDelayInMillis( theExpiresIn ) );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void dispose() {
		_timer.cancel();
		super.dispose();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * We wait for about 85% of the expiration time to be inside the refresh
	 * time-slot.
	 * 
	 * @param aExpiresInSecs The expiration time in seconds.
	 * @return The calculated expiration time in milliseconds.
	 */
	private long toDelayInMillis( Integer aExpiresInSecs ) {
		Double theTimeout = ((double) aExpiresInSecs) * ((double) 1000) * 0.85;
		long theDelay = theTimeout.longValue();
		return theDelay;
	}

	/**
	 * Retrieves an {@link OauthToken} by applying the grant type "password" to
	 * the given URL.
	 * 
	 * @param aUrl The URL from which to get the {@link OauthToken}
	 * @param aHttpRestClient The {@link HttpRestClient} to use for querying the
	 *        {@link OauthToken}.
	 * @param aUserName The user's name for "password" grant type.
	 * @param aUserPassword The user's password for "password" grant type.
	 * @return The {@link OauthToken}.
	 * @throws MalformedURLException thrown in case of a malformed URL.
	 * @throws HttpStatusException thrown in case a HTTP response was of an
	 *         erroneous status.
	 */
	private static OauthToken toOauthToken( String aUrl, HttpRestClient aHttpRestClient, String aClientId, String aClientSecret, String aUserName, String aUserPassword ) throws MalformedURLException, HttpStatusException {
		RestRequestBuilder theRequestBuilder = aHttpRestClient.buildPost( aUrl );
		theRequestBuilder.getHeaderFields().putAcceptTypes( MediaType.APPLICATION_JSON );

		// |--> Client-ID + Client-Secret = Form-Fields:
		if ( aClientId != null && aClientId.length() != 0 ) {
			FormFields theFormFields = new FormFieldsImpl();
			theFormFields.put( OauthField.CLIENT_ID.getName(), aClientId );
			if ( aClientSecret != null && aClientSecret.length() != 0 ) theFormFields.put( OauthField.CLIENT_SECRET.getName(), aClientSecret );
			if ( aUserName != null && aUserName.length() != 0 ) theFormFields.put( OauthField.USERNAME.getName(), aUserName );
			if ( aUserPassword != null && aUserPassword.length() != 0 ) theFormFields.put( OauthField.PASSWORD.getName(), aUserPassword );
			theFormFields.put( OauthField.GRANT_TYPE.getName(), GrantType.PASSWORD.getValue() );
			theRequestBuilder.setRequest( theFormFields );
		}
		// Client-ID + Client-Secret: Form-Fields <--|
		// |--> Otherwise = Basic-Auth in the HTTP-Header:
		else {
			BasicAuthCredentials theCredentials = new BasicAuthCredentialsImpl( aUserName, aUserPassword );
			theRequestBuilder.getHeaderFields().putBasicAuthCredentials( theCredentials );
		}
		// Otherwise = Basic-Auth in the HTTP-Header <--|

		RestResponse response = theRequestBuilder.toRestResponse();
		if ( response.getHttpStatusCode().isErrorStatus() ) {
			throw response.getHttpStatusCode().toHttpStatusException( response.getHttpBody() );
		}
		OauthToken theToken = new OauthTokenImpl( response.getResponse() );
		return theToken;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@link TimerTask} for refreshing the access token.
	 */
	private class RefreshTask extends TimerTask {

		@Override
		public void run() {
			if ( !isDisposed() ) {
				synchronized ( OauthTokenHandler.this ) {
					if ( !_restClient.isConnectionOpened() ) {
						dispose();
					}
					else {
						Integer theBeforeExpiresIn = _expiresIn;
						for ( int i = 0; i < MAX_REFRESH_TRIES; i++ ) {
							try {
								if ( getRefreshToken() == null ) {
									// --------------------------------------------
									// Will never get new token again: Stop dameon!
									// --------------------------------------------
									_timer.cancel();
									// --------------------------------------------------------------------------
									// Though the access token might still be valid:  Do not dispose! Just return
									// --------------------------------------------------------------------------
									return;
								}
								RestRequestBuilder theBuilder = _restClient.buildPost( _url );
								theBuilder.getHeaderFields().putAcceptTypes( MediaType.APPLICATION_JSON );
								theBuilder.getHeaderFields().putContentType( MediaType.APPLICATION_X_WWW_FORM_URLENCODED );
								FormFields theFormFields = new FormFieldsImpl();
								theFormFields.put( OauthField.GRANT_TYPE.getName(), GrantType.REFRESH_TOKEN.getValue() );
								theFormFields.put( OauthField.REFRESH_TOKEN.getName(), getRefreshToken() );
								if ( getScope() != null ) theFormFields.put( OauthField.SCOPE.getName(), getScope() );
								RestResponse theResponse = theBuilder.toRestResponse();
								if ( theResponse.getHttpStatusCode().isSuccessStatus() ) {
									HttpBodyMap theBody = theResponse.getResponse();

									// "access_token":
									if ( !theBody.containsKey( OauthField.ACCESS_TOKEN.getPath() ) ) {
										break;
									}
									// "token_type":
									if ( !theBody.containsKey( OauthField.TOKEN_TYPE.getPath() ) ) {
										_tokenType = theBody.get( OauthField.TOKEN_TYPE.getPath() );
										break;
									}

									// Copy properties:
									fromHttpBodyMap( theBody );

									// "expires_in":
									if ( theBeforeExpiresIn != null && !theBeforeExpiresIn.equals( _expiresIn ) ) {
										Integer theExpiresIn = _expiresIn;
										if ( theExpiresIn == null || theExpiresIn == -1 ) {
											theExpiresIn = DEFAULT_EXPIRES_IN;
										}
										_timer.cancel();
										_timer = new Timer( true );
										_timer.schedule( _timerTask, toDelayInMillis( theExpiresIn ) );
									}

									return;
								}
							}
							catch ( MalformedURLException | HttpResponseException e ) {
								if ( i < MAX_REFRESH_TRIES ) {
									try {
										Thread.sleep( MAX_RETRY_SLEEP_TIME );
									}
									catch ( InterruptedException ignore ) {}
								}
							}
						}
						dispose();
					}
				}
			}
		}
	}
}
