// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import org.refcodes.net.HttpClientRequest;
import org.refcodes.net.HttpClientRequestImpl;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.HttpStatusCode;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.Url;
import org.refcodes.net.UrlBuilderImpl;
import org.refcodes.textual.VerboseTextBuilderImpl;

/**
 * The implementation of the {@link RestCallerBuilder} interface as good old
 * POJO for use by different {@link RestClient} implementations.
 */
public class RestRequestBuilderImpl extends HttpClientRequestImpl implements RestRequestBuilder {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private RestClient _restClient;

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new rest request builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aHeaderFields the Header-Fields
	 * @param aRequest the request
	 * @param aRestClient the rest client
	 */
	protected RestRequestBuilderImpl( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestClient aRestClient ) {
		this( aHttpMethod, aUrl instanceof UrlBuilder ? aUrl : new UrlBuilderImpl( aUrl ), aHeaderFields, aRequest, HttpClientRequest.DEFAULT_REDIRECT_DEPTH, aRestClient );
	}

	/**
	 * Instantiates a new rest request builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aRestClient the rest client
	 */
	protected RestRequestBuilderImpl( HttpMethod aHttpMethod, Url aUrl, RestClient aRestClient ) {
		this( aHttpMethod, aUrl, null, null, aRestClient );
	}

	/**
	 * Instantiates a new rest request builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aRequest the request
	 * @param aRestClient the rest client
	 */
	protected RestRequestBuilderImpl( HttpMethod aHttpMethod, Url aUrl, Object aRequest, RestClient aRestClient ) {
		this( aHttpMethod, aUrl, null, aRequest, aRestClient );
	}

	/**
	 * Instantiates a new rest request builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aRedirectDepth The redirect depth provides the count of
	 *        HTTP-Request and HTTP-Response cycles where the response
	 *        represents a redirect as of
	 *        {@link HttpStatusCode#isRedirectStatus()}. A value of -1
	 *        represents the default behavior, e.g. using
	 *        {@link java.net.HttpURLConnection}'s redirection means.
	 * @param aRestClient the rest client
	 */
	protected RestRequestBuilderImpl( HttpMethod aHttpMethod, Url aUrl, int aRedirectDepth, RestClient aRestClient ) {
		this( aHttpMethod, aUrl, null, null, aRedirectDepth, aRestClient );
	}

	/**
	 * Instantiates a new rest request builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aRequest the request
	 * @param aRedirectDepth The redirect depth provides the count of
	 *        HTTP-Request and HTTP-Response cycles where the response
	 *        represents a redirect as of
	 *        {@link HttpStatusCode#isRedirectStatus()}. A value of -1
	 *        represents the default behavior, e.g. using
	 *        {@link java.net.HttpURLConnection}'s redirection means.
	 * @param aRestClient the rest client
	 */
	protected RestRequestBuilderImpl( HttpMethod aHttpMethod, Url aUrl, Object aRequest, int aRedirectDepth, RestClient aRestClient ) {
		this( aHttpMethod, aUrl, null, aRequest, aRedirectDepth, aRestClient );
	}

	/**
	 * Instantiates a new rest request builder impl.
	 *
	 * @param aHttpMethod the http method
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * @param aHeaderFields the Header-Fields
	 * @param aRequest the request
	 * @param aRedirectDepth The redirect depth provides the count of
	 *        HTTP-Request and HTTP-Response cycles where the response
	 *        represents a redirect as of
	 *        {@link HttpStatusCode#isRedirectStatus()}. A value of -1
	 *        represents the default behavior, e.g. using
	 *        {@link java.net.HttpURLConnection}'s redirection means.
	 * @param aRestClient the rest client
	 */
	protected RestRequestBuilderImpl( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, int aRedirectDepth, RestClient aRestClient ) {
		super( aHttpMethod, aUrl instanceof UrlBuilder ? aUrl : new UrlBuilderImpl( aUrl ), aHeaderFields, aRequest, aRedirectDepth, aRestClient );
		_restClient = aRestClient;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Url.UrlBuilder getUrl() {
		return (Url.UrlBuilder) _url;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setUrl( Url aUrl ) {
		_url = aUrl;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HttpMethod getHttpMethod() {
		return _httpMethod;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHttpMethod( HttpMethod aHttpMethod ) {
		_httpMethod = aHttpMethod;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return getClass().getSimpleName() + "(" + _httpMethod + ": " + _url.toHttpUrl() + "?" + new VerboseTextBuilderImpl().withElements( _url.getQueryFields() ).toString() + ")@" + hashCode();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHeaderFields( RequestHeaderFields aHeaderFields ) {
		_headerFields = aHeaderFields;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RestResponse toRestResponse() throws HttpResponseException {
		return _restClient.doRequest( getHttpMethod(), getUrl(), getHeaderFields(), (Object) getRequest(), getRedirectDepth() );
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

}
