// /////////////////////////////////////////////////////////////////////////////
// REFCODES.ORG
// /////////////////////////////////////////////////////////////////////////////
// This code is copyright (c) by Siegfried Steiner, Munich, Germany and licensed
// under the following (see "http://en.wikipedia.org/wiki/Multi-licensing")
// licenses:
// -----------------------------------------------------------------------------
// GNU General Public License, v3.0 ("http://www.gnu.org/licenses/gpl-3.0.html")
// -----------------------------------------------------------------------------
// Apache License, v2.0 ("http://www.apache.org/licenses/LICENSE-2.0")
// -----------------------------------------------------------------------------
// Please contact the copyright holding author(s) of the software artifacts in
// question for licensing issues not being covered by the above listed licenses,
// also regarding commercial licensing models or regarding the compatibility
// with other open source licenses.
// /////////////////////////////////////////////////////////////////////////////

package org.refcodes.rest;

import java.net.MalformedURLException;

import org.refcodes.data.Scheme;
import org.refcodes.net.BaseUrlAccessor;
import org.refcodes.net.FormFields;
import org.refcodes.net.HttpClientRequest;
import org.refcodes.net.HttpMethod;
import org.refcodes.net.HttpResponseException;
import org.refcodes.net.HttpStatusCode;
import org.refcodes.net.RequestHeaderFields;
import org.refcodes.net.Url;
import org.refcodes.net.UrlImpl;

/**
 * Helper interface to keep the huge amount of convenience methods under
 * control.
 */
public interface RestRequestClient {

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath ) {
		UrlImpl theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, Object aRequest ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aUrl ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aUrl );
		// In case of relative path, use base URL (if applicable) |-->
		if ( theUrl.getHost() == null && theUrl.getScheme() == null && this instanceof BaseUrlAccessor ) {
			BaseUrlAccessor theBaseUrlAccessor = (BaseUrlAccessor) this;
			Url theBaseUrl = theBaseUrlAccessor.getBaseUrl();
			if ( theBaseUrl != null ) {
				theUrl = new UrlImpl( theBaseUrlAccessor.getBaseUrl(), aUrl );
			}
		}
		// <--| In case of relative path, use base URL (if applicable)
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The local targetd locator.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The targeted locator.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aUrl, Object aRequest ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aUrl );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aUrl );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return buildRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return buildRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @return The {@link RestRequestBuilder} which is used to prepare and
	 *         synchronously issue the request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Url aUrl ) {
		return buildRequest( aHttpMethod, aUrl, (RequestHeaderFields) null, null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields ) {
		return buildRequest( aHttpMethod, aUrl, aHeaderFields, null );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	default RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) {
		return buildRequest( aHttpMethod, aUrl, aHeaderFields, aRequest, HttpClientRequest.DEFAULT_REDIRECT_DEPTH );
	}

	/**
	 * Prepares a request builder with the possible attributes and returns the
	 * according request's {@link RestRequestBuilder} instance. Invoking
	 * {@link RestRequestBuilder#toRestResponse} starts the request and
	 * synchronously returns the response.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestRequestBuilder} at this request.
	 */
	RestRequestBuilder buildRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, int aRedirectCount );

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aScheme, aHost, aPath, aQueryFields, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aScheme The {@link Scheme} (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Scheme aScheme, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aScheme, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @return The {@link RestResponse} as of your request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aUrl ) throws HttpResponseException, MalformedURLException {
		Url theUrl = new UrlImpl( aUrl );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields ) throws HttpResponseException, MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return doRequest( aHttpMethod, theUrl, null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException, MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 *
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aLocator The targeted locator, usually the part of the URL till
	 *        (excluding) the query fields and the fragment.
	 * 
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 * 
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aLocator, FormFields aQueryFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aLocator, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aUrl, Object aRequest ) throws HttpResponseException, MalformedURLException {
		Url theUrl = new UrlImpl( aUrl );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aUrl, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aUrl );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * @return The {@link RestResponse} as of your request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields ) throws HttpResponseException, MalformedURLException {
		Url theUrl = new UrlImpl( aUrl );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException, MalformedURLException {
		return doRequest( aHttpMethod, aUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		Url theUrl = new UrlImpl( aUrl );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aUrl, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( aHttpMethod, aUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 * 
	 * @param aUrl The targeted URL locating the addressed resource.
	 * 
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aUrl, RestResponseObserver aResponseObserver ) throws MalformedURLException {
		return doRequest( aHttpMethod, aUrl, (RequestHeaderFields) null, (Object) null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPort The port to be used when connecting to the host.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, int aPort, String aPath, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPort, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aQueryFields The Query-Fields to be used for the HTTP
	 *        Query-String.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, FormFields aQueryFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath, aQueryFields );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 *
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @return The {@link RestResponse} as of your request.
	 *
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, aRequest, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aProtocol The protocol (e.g. HTTP or HTTPS) to be used for the
	 *        destination URL.
	 *
	 * @param aHost The host to which the destination URL is to point to.
	 *
	 * @param aPath The path on the host to which the base destination URL is to
	 *        point to.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, String aProtocol, String aHost, String aPath, RestResponseObserver aResponseObserver ) {
		Url theUrl = new UrlImpl( aProtocol, aHost, aPath );
		return doRequest( aHttpMethod, theUrl, (RequestHeaderFields) null, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Url aUrl ) throws HttpResponseException {
		return doRequest( aHttpMethod, aUrl, (RequestHeaderFields) null, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Url aUrl, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aUrl, null, aRequest );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Url aUrl, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aUrl, (RequestHeaderFields) null, aRequest, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 * 
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields ) throws HttpResponseException {
		return doRequest( aHttpMethod, aUrl, aHeaderFields, (Object) null );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	default RestResponse doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest ) throws HttpResponseException {
		return doRequest( aHttpMethod, aUrl, aHeaderFields, aRequest, HttpClientRequest.DEFAULT_REDIRECT_DEPTH );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @param aRedirectDepth The redirect depth provides the count of
	 *        HTTP-Request and HTTP-Response cycles where the response
	 *        represents a redirect as of
	 *        {@link HttpStatusCode#isRedirectStatus()}. A value of -1
	 *        represents the default behavior, e.g. using
	 *        {@link java.net.HttpURLConnection}'s redirection means.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 */
	RestResponse doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, int aRedirectDepth ) throws HttpResponseException;

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aUrl, (RequestHeaderFields) aHeaderFields, null, aResponseObserver );
	}

	/**
	 * Sends a request with the possible attributes and returns the according
	 * request's {@link RestResponse} instance synchronously.
	 * 
	 * @param aRequest The {@link RestRequest} encapsulating all necessary
	 *        attributes to issue the request.
	 * 
	 * @return The {@link RestResponse} for this request.
	 * 
	 * @throws HttpResponseException thrown in case of some unexpected response.
	 * 
	 * @throws MalformedURLException in case the provided URL is considered
	 *         being malformed.
	 */
	default RestResponse doRequest( RestRequest aRequest ) throws HttpResponseException, MalformedURLException {
		return doRequest( aRequest.getHttpMethod(), aRequest.getUrl(), aRequest.getHeaderFields(), (Object) aRequest.getRequest() );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	default RestCallerBuilder doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, RestResponseObserver aResponseObserver ) {
		return doRequest( aHttpMethod, aUrl, aHeaderFields, aRequest, HttpClientRequest.DEFAULT_REDIRECT_DEPTH, aResponseObserver );
	}

	/**
	 * Creates a request with the possible attributes and returns the according
	 * {@link RestCallerBuilder} instance used for the request.
	 * 
	 * The HTTP-Request is actually sent not earlier than you calling
	 * {@link RestCallerBuilder#open()} on the returned
	 * {@link RestCallerBuilder} as the {@link RestCallerBuilder} still may be
	 * modified after invoking this method!
	 * 
	 * @param aHttpMethod The HTTP-Method for the request.
	 *
	 * @param aUrl The {@link Url} from which to take the URL specific data.
	 *
	 * @param aHeaderFields The HTTP-Header's fields to be used for the request.
	 *
	 * @param aRequest The body to be sent with the request (e.g. when doing a
	 *        {@link HttpMethod#POST} request.
	 * 
	 * @param aRedirectDepth The redirect depth provides the count of
	 *        HTTP-Request and HTTP-Response cycles where the response
	 *        represents a redirect as of
	 *        {@link HttpStatusCode#isRedirectStatus()}. A value of -1
	 *        represents the default behavior, e.g. using
	 *        {@link java.net.HttpURLConnection}'s redirection means.
	 *
	 * @param aResponseObserver The listener processing a response targeted at
	 *        this {@link RestCallerBuilder}.
	 *
	 * @return The {@link RestCallerBuilder} which is used by the request.
	 */
	RestCallerBuilder doRequest( HttpMethod aHttpMethod, Url aUrl, RequestHeaderFields aHeaderFields, Object aRequest, int aRedirectDepth, RestResponseObserver aResponseObserver );

}
