#!/bin/bash

# ------------------------------------------------------------------------------
# Set script specific data:
# ------------------------------------------------------------------------------
SCRIPT_NAME="$(basename $0 .sh)"
SCRIPT_PATH="$(dirname $0)"
# ------------------------------------------------------------------------------
# Set default values overwritten by "application.properties:
# ------------------------------------------------------------------------------
# DEBUG_PORT=5005
APP_NAME="${SCRIPT_NAME//-}"
APP_LOG_DIR="logs"
# APP_LOG_NAME="${APP_NAME}"
APP_LOG_NAME="application"
APP_PID_DIR="pid"
# APP_PID_NAME="${APP_NAME}"
APP_PID_NAME="application"

# ------------------------------------------------------------------------------
# Prepare potential property file locations:
# ------------------------------------------------------------------------------
APP_PROPS_NAME="application.properties"

# ------------------------------------------------------------------------------
# Define functions:
# ------------------------------------------------------------------------------

function print_banner {
    which figlet > /dev/null 2> /dev/null
    if (( $? == 0 )); then
        figlet -w 120 ${APP_NAME}
    else
        which banner > /dev/null 2> /dev/null
        if (( $? == 0 )); then
            banner ${APP_NAME}
        else
            echo '>>>>>>>>>>'" [ ${APP_NAME^^} ] "'<<<<<<<<<<'
            echo "(please install 'figlet' or 'banner' for a nice banner)"
        fi
   fi
}

function print_base_path {
    if [[ "${SCRIPT_PATH}" == */target ]] ; then
        echo "${SCRIPT_PATH}"
    elif [ -f "${SCRIPT_PATH}/${APP_PROPS_NAME}" ] ; then
        echo "${SCRIPT_PATH}"
    else
        echo "${SCRIPT_PATH}/.."
    fi
    return 0
}

function print_jar_path {
    length=${#APP_NAME}
    for f in ${jarPaths[@]}; do
         ls -t ${f}/${APP_NAME}*.jar > /dev/null 2> /dev/null
        if (( $? == 0 )) ; then
            echo $(ls -t ${f}/${APP_NAME}*.jar | head -1)
            return 0
        fi
        for (( i=1 ; i < $length ; i++ )) ; do
            JAR_NAME=${APP_NAME:0:$i}-${APP_NAME:$i}
            ls -t ${f}/$JAR_NAME*.jar > /dev/null 2> /dev/null
            if (( $? == 0 )) ; then
                echo $(ls -t ${f}/$JAR_NAME*.jar | head -1)
                return 0
            fi
        done
    done
    return 1
}

function on_jar_path_not_found {
    echo "No JAR files to execute found in neither of the below directories:"
    for f in ${jarPaths[@]}; do
        echo "  $(readlink -f $f)"
    done
}

function exit_on_jar_path_not_found {
    on_jar_path_not_found
    echo "Aborting!"
    exit 1
}

function print_app_props_file {
    for f in ${appPropsPaths[@]}; do
        if [ -f "${f}/${APP_PROPS_NAME}" ] ; then
            echo ${f}/${APP_PROPS_NAME}
            return 0
        fi
    done
   return 1
}

function print_app_props_path {
    for f in ${appPropsPaths[@]}; do
        if [ -d "${f}" ] ; then
            echo ${f}
            return 0
        fi
    done
   return 1
}

function on_app_props_file_not_found {
    echo "No property file \"${APP_PROPS_NAME}\" found in neither of the below locations:"
    for f in ${appPropsPaths[@]}; do
        echo "  $(readlink -f $f)"
    done
}

function exit_on_app_props_file_not_found {
    on_app_props_file_not_found
    echo "Aborting!"
    exit 1
}

function check_if_pid_file_exists {
    if [ -f ${pidFile} ] ; then
        return 0
    else
    	return 1
    fi
}

function check_if_process_is_running {
    if ps -p $(print_process) > /dev/null ; then
        return 0
    else
        return 1
    fi
}

function print_process {
    echo $(<"${pidFile}")
}

function write_process {
    echo $$ > "${pidFile}"
}

function print_settings {
    # --- application.properties ---
    appPropsFile=$(print_app_props_file)
     if [ $? -ge 1  ] ; then
        echo -e "\e[31m[ FAIL ]\e[0m <-- Loading properties \"${APP_PROPS_NAME}\""
        on_app_props_file_not_found
    else
        echo -e "\e[32m[  OK  ]\e[0m <-- Loading properties from \"$(readlink -f ${appPropsFile})\""
    fi
    # --- PID ---
    if [ -d ${pidPath} ] ; then
        echo -e "\e[32m[  OK  ]\e[0m <-- Writing PID file to \"$(readlink -f ${pidPath})/${pidName}\""
    else
        echo -e "\e[33m[CREATE]\e[0m <-- Creating PID file at \"${pidPath}/${pidName}\""
    fi
    # --- STD OUT ---
    if [ -d ${logPath} ] ; then
        echo -e "\e[32m[  OK  ]\e[0m <-- Logging STD out to \"$(readlink -f ${logPath})/${stdLogName}\""
    else
        echo -e "\e[33m[CREATE]\e[0m <-- Logging STD out to \"${logPath}/${stdLogName}\""
    fi
    # --- STD_ERR ---
    if [ -d ${logPath} ] ; then
        echo -e "\e[32m[  OK  ]\e[0m <-- Logging ERR out to \"$(readlink -f ${logPath})/${errLogName}\""
    else
        echo -e "\e[33m[CREATE]\e[0m <-- Logging ERR out to \"${logPath}/${errLogName}\""
    fi
    # --- JAR & PORT ---
    jarPath="$(print_jar_path)"
    if [ $? -ge 1  ] ; then
        echo -e "\e[31m[ FAIL ]\e[0m <-- Executing binary on PORT [${SERVER_PORT}]"
        on_jar_path_not_found
    else
        echo -e "\e[32m[  OK  ]\e[0m <-- Executing binary at \"$(readlink -f ${jarPath})\" on PORT [${SERVER_PORT}]"
    fi
    # --- Debug ---
    if [ -n ${DEBUG_PORT} ] ; then
        echo -e "\e[32m[  OK  ]\e[0m <-- Using debug PORT [${DEBUG_PORT}] when starting in debug mode"
    else
        echo -e "\e[31m[ FAIL ]\e[0m <-- No debug PORT specified for starting in debug mode"
    fi
    # --- JMX ---
    if [ -n ${MANAGEMENT_PORT} ] ; then
        echo -e "\e[32m[  OK  ]\e[0m <-- Using management PORT [${MANAGEMENT_PORT}] for JMX HTTP monitoring and management"
    else
        echo -e "\e[31m[ FAIL ]\e[0m <-- No management PORT specified  for JMX HTTP monitoring and management"
    fi
}
function print_jmx_help {
    echo "Usage: $0 {autoconfig|beans|configprops|dump|env|health|info|metrics|mappings|shutdown|trace}"
    echo "      beans: Displays a complete list of all the Spring Beans in your application."
    echo "configprops: Displays a collated list of all @ConfigurationProperties."
    echo "       dump: Performs a thread dump."
    echo "        env: Exposes properties from Spring’s ConfigurableEnvironment."
    echo "     health: Shows application health information (defaulting to a simple ‘OK’ message)."
    echo "       info: Displays arbitrary application info."
    echo "    metrics: Shows ‘metrics’ information for the current application."
    echo "   mappings: Displays a collated list of all @RequestMapping paths."
    echo "   shutdown: Allows the application to be gracefully shutdown (not enabled by default)."
    echo "      trace: Displays trace information (by default the last few HTTP requests)."
}

function print_help {
    echo "Usage: $0 {start [tail]|debug [tail]|stop [tail]|graceful-stop [tail]|restart [tail]|graceful-restart [tail]|tail|status|settings|jmx}"
}

# ------------------------------------------------------------------------------
# Determine base path:
# ------------------------------------------------------------------------------
BASE_PATH=$(print_base_path)

# ------------------------------------------------------------------------------
# Load properties:
# ------------------------------------------------------------------------------
appPropsPaths=("${BASE_PATH}" "${BASE_PATH}/config" "${BASE_PATH}/etc")
APP_PROPS_PATH=$(print_app_props_file)
if [ $? -ge 1  ] ; then
    if [ "$1" != "test" ] ; then
        print_banner
        exit_on_app_props_file_not_found
    fi
else
    . ${APP_PROPS_PATH}
fi

# ------------------------------------------------------------------------------
# Prepare internal variables:
# ------------------------------------------------------------------------------
jarPaths=("${BASE_PATH}/bin" "${BASE_PATH}/libs" "${BASE_PATH}")
now="$(date +%F)"
logPath="${BASE_PATH}/${APP_LOG_DIR}"
stdLogName="${APP_LOG_NAME}.${now}.out"
errLogName="${APP_LOG_NAME}.${now}.err"
stdLogFile="${logPath}/${stdLogName}"
errLogFile="${logPath}/${errLogName}"
pidPath="${BASE_PATH}/${APP_PID_DIR}"
pidName="${APP_PID_NAME}.pid"
pidFile="${pidPath}/${pidName}"
SPRING_OPTS="-DLOG_FILE=$stdLogFile --spring.config.location=$(print_app_props_file)"

# ------------------------------------------------------------------------------
# Do the job:
# ------------------------------------------------------------------------------

print_banner
case "$1" in
  status)
    if ! check_if_pid_file_exists ; then 
    	echo "PID file not found: ${pidFile}"
        exit 1
    fi
    if check_if_process_is_running ; then
      echo -e "\e[32mProcess [$(print_process)] is running.\e[0m"
    else
      echo -e "\e[33mProcess $(print_process) not running.\e[0m"
    fi
    ;;
  stop)
    if ! check_if_pid_file_exists ; then 
    	echo -e "\e[31mPID file not found: ${pidFile}\e[0m"
        exit 1
    fi
    if ! check_if_process_is_running ; then
      echo -e "\e[33mProcess $(print_process) already stopped.\e[0m"
      exit 0
    fi
    # kill -TERM $(print_process)
    kill -TERM $(print_process)
    echo -ne "Waiting for process $(print_process) to stop"
    NOT_KILLED=1
    for i in {1..20}; do
      if check_if_process_is_running ; then
        echo -ne "."
        sleep 1
      else
        NOT_KILLED=0
      fi
    done
    echo
    if [ $NOT_KILLED = 1 ] ; then
      echo -e "\e[31mCannot kill process [$(print_process)]!\e[0m"
      exit 1
    fi
    echo -e "\e[32mProcess [$(print_process)] stopped.\e[0m"
    ;;
  graceful-stop|stop-graceful)
    if check_if_pid_file_exists ; then
		if ! check_if_process_is_running ; then
		  echo -e "\e[33mProcess [$(print_process)] already stopped.\e[0m"
		  exit 0
		fi
	fi

    echo -ne "Stopping via JMX shutdown: "
    curl -X POST localhost:${MANAGEMENT_PORT}/shutdown
    echo ""
    if check_if_pid_file_exists ; then 
		echo -ne "Waiting for process [$(print_process)] to stop gracefully"
		NOT_KILLED=1
		for i in {1..20}; do
		  if check_if_process_is_running ; then
			echo -ne "."
			sleep 1
		  else
			NOT_KILLED=0
		  fi
		done
		echo
		if [ $NOT_KILLED = 1 ] ; then
		  echo e "\e[31mCannot stop process [$(print_process)] gracefully!\e[0m"
		  exit 1
		fi
	fi
    echo -e "\e[32mProcess [$(print_process)] stopped gracefully.\e[0m"
    ;;
  start|debug|suspend-debug)
    if [ -f ${pidFile} ] && check_if_process_is_running ; then
      echo -e "\e[33mProcess [$(print_process)] already running.\e[0m"
  	  exit 0
    fi
    if [ -f ${pidFile} ]; then
        rm ${pidFile}
    fi
    mkdir -p "${pidPath}"
    # mkdir -p "${logPath}"
    JAR_FILE="$(print_jar_path)"
    if [ $? -ge 1  ] ; then
        exit_on_jar_path_not_found
        echo -e "\e[31mAborting!\e[0m"
        exit 1
    fi
    if [ "$1" == "debug" ] || [ "$1" == "suspend-debug" ] ; then
    	startMsgSuffix=" in DEBUG mode using PORT [$DEBUG_PORT]"
    	suspend="n"
    	if [ "$1" == "suspend-debug" ] ; then
    		suspend="y"
    		startMsgSuffix="${startMsgSuffix}: Waiting for your remote debugger to connect..."
		else
			startMsgSuffix="${startMsgSuffix}." 
    	fi
        javaArgs="-agentlib:jdwp=transport=dt_socket,server=y,suspend=${suspend},address=$DEBUG_PORT "
        
    else
        javaArgs=""
        startMsgSuffix="."
    fi
    mkdir -p "${logPath}"
    nohup java -jar $javaArgs $JAR_FILE $SPRING_OPTS > $stdLogFile 2> $errLogFile & echo $! > ${pidFile}
    echo -ne "Starting process for $(readlink -f $(print_jar_path))"
    for i in {1..20}; do
      if  [ -f ${pidFile} ] ; then
        :
      else
        echo -ne "."
        sleep 1
      fi
    done
    echo
    echo -e "\e[32mProcess [$(print_process)] started$startMsgSuffix\e[0m"
    if [ "$2" == "tail" ]  ; then
        tail -f ${errLogFile} -f ${stdLogFile}
    fi
    ;;
  restart)
    $0 stop
    if [ $? = 1 ]
    then
      exit 1
    fi
    $0 start
    ;;
  graceful-restart|restart-graceful)
    $0 graceful-stop
    if [ $? = 1 ]
    then
      exit 1
    fi
    $0 start
    ;;
  settings)
    print_settings
    ;;
  tail)
    tail -f ${errLogFile} -f ${stdLogFile}
    ;;
  autoconfig)
    curl localhost:${MANAGEMENT_PORT}/autoconfig
    ;;
  beans)
    curl localhost:${MANAGEMENT_PORT}/beans
    ;;
  configprops)
    curl localhost:${MANAGEMENT_PORT}/configprops
    ;;
  dump)
    curl localhost:${MANAGEMENT_PORT}/dump
    ;;
  env)
    curl localhost:${MANAGEMENT_PORT}/env
    ;;
  health)
    curl localhost:${MANAGEMENT_PORT}/health
    ;;
  info)
    curl localhost:${MANAGEMENT_PORT}/info
    ;;
  metrics)
    curl localhost:${MANAGEMENT_PORT}/metrics
    ;;
  mappings)
    curl localhost:${MANAGEMENT_PORT}/mappings
    ;;
  shutdown)
    curl -X POST localhost:${MANAGEMENT_PORT}/shutdown
    ;;
  trace)
    curl localhost:${MANAGEMENT_PORT}/trace
    ;;
  jmx)
    print_jmx_help
    exit 1
    ;;
  *)
    print_help
    exit 1
esac
exit 0