/*
 * FixedProducer.java February 2007
 *
 * Copyright (C) 2001, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;
import java.nio.ByteBuffer;

/**
 * The <code>FixedProducer</code> object produces content without any
 * encoding, but limited to a fixed number of bytes. This is used if
 * the length of the content being delivered is know beforehand. It 
 * will simply count the number of bytes being send and signal the
 * server kernel that the next request is ready to read once all of
 * the bytes have been sent to the client.
 *
 * @author Niall Gallagher
 *
 * @see org.simpleframework.http.core.FixedConsumer
 */ 
class FixedProducer implements Producer{

   /**
    * This is the monitor used to notify the initiator of events.
    */ 
   private Monitor monitor;

   /**
    * This is the underlying sender used to deliver the raw data.
    */ 
   private Sender sender;

   /**
    * This is the number of bytes that have been sent so far.
    */ 
   private int count;
   
   /**
    * This is the number of bytes this producer is limited to.
    */ 
   private int limit;
   
   /**
    * Constructor for the <code>FixedProducer</code> object. This is
    * used to create a producer that will count the number of bytes
    * that are sent over the pipeline, once all bytes have been sent
    * this will signal that the next request is ready to read.
    *
    * @param sender this is used to send to the underlying transport
    * @param monitor this is used to deliver signals to the kernel
    * @param limit this is used to limit the number of bytes sent
    */ 
   public FixedProducer(Sender sender, Monitor monitor, int limit) {
      this.monitor = monitor;   
      this.sender = sender;
      this.limit = limit;
   }
  
   /**
    * This method is used to encode the provided array of bytes in
    * a HTTP/1.1 complaint format and sent it to the client. Once
    * the data has been encoded it is handed to the transport layer
    * within the server, which may choose to buffer the data if the
    * content is too small to send efficiently or if the socket is
    * not write ready.
    *
    * @param array this is the array of bytes to send to the client
    */        
   public void produce(byte[] buf) throws IOException {
      produce(buf, 0, buf.length);
   }
   
   /**
    * This method is used to encode the provided array of bytes in
    * a HTTP/1.1 complaint format and sent it to the client. Once
    * the data has been encoded it is handed to the transport layer
    * within the server, which may choose to buffer the data if the
    * content is too small to send efficiently or if the socket is
    * not write ready.
    *
    * @param array this is the array of bytes to send to the client
    * @param off this is the offset within the array to send from
    * @param len this is the number of bytes that are to be sent
    */       
   public void produce(byte[] buf, int off, int len) throws IOException {
      ByteBuffer buffer = ByteBuffer.wrap(buf, off, len);
      
      if(len > 0) {
         produce(buffer);
      }  
   }
   
   /**
    * This method is used to encode the provided buffer of bytes in
    * a HTTP/1.1 compliant format and sent it to the client. Once
    * the data has been encoded it is handed to the transport layer
    * within the server, which may choose to buffer the data if the
    * content is too small to send efficiently or if the socket is
    * not write ready.
    *
    * @param buffer this is the buffer of bytes to send to the client
    */         
   public void produce(ByteBuffer buffer) throws IOException {
      int mark = buffer.position();
      int size = buffer.limit();
      
      if(mark > size) {
         throw new ProducerException("Buffer position greater than limit");
      }
      produce(buffer, 0, size - mark);
   }

   /**
    * This method is used to encode the provided buffer of bytes in
    * a HTTP/1.1 compliant format and sent it to the client. Once
    * the data has been encoded it is handed to the transport layer
    * within the server, which may choose to buffer the data if the
    * content is too small to send efficiently or if the socket is
    * not write ready.
    *
    * @param buffer this is the buffer of bytes to send to the client
    * @param off this is the offset within the buffer to send from
    * @param len this is the number of bytes that are to be sent
    */          
   public void produce(ByteBuffer buffer, int off, int len) throws IOException {
      int size = Math.min(len, limit - count);          
      
      try {
         if(monitor.isClosed()) {
            throw new ProducerException("Response content complete");
         }
         sender.send(buffer, off, size);
         
         if(count + size == limit) {
            monitor.ready(sender);
         }        
      } catch(Exception cause) {
         if(sender != null) {
            monitor.error(sender);
         }
         throw new ProducerException("Error sending response", cause);
      }
      count += size;
   }
   
   /**
    * This method is used to flush the contents of the buffer to 
    * the client. This method will block until such time as all of
    * the data has been sent to the client. If at any point there
    * is an error sending the content an exception is thrown.    
    */    
   public void flush() throws IOException {
      try {
         if(!monitor.isClosed()) {
            sender.flush();
         }
      } catch(Exception cause) {
         if(sender != null) {
            monitor.error(sender);
         }
         throw new ProducerException("Error flushing", cause);
      }
   }
   
   /**
    * This is used to signal to the producer that all content has 
    * been written and the user no longer needs to write. This will
    * either close the underlying transport or it will notify the
    * monitor that the response has completed and the next request
    * can begin. This ensures the content is flushed to the client.
    */   
   public void close() throws IOException {
      if(!monitor.isClosed()) {
         if(count < limit) {
            monitor.error(sender);
         } else {
            monitor.ready(sender);
         }
      }
   }
}
