/*
 * LargeConsumer.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;

/**
 * The <code>LargeConsumer</code> object is used to create a consumer
 * that is used to consume large entity bodies. Typically a large
 * body will be one that is delivered as part of a multipart upload
 * or as a large form POST. The task of the large consumer is to
 * consume all the bytes for the body, and reset the cursor after the
 * last byte that has been send with the body. This ensures that the
 * next character read from the cursor is the first character of a
 * HTTP header within the pipeline.
 *
 * @author Niall Gallagher
 */ 
abstract class LargeConsumer extends BodyConsumer { 

   /**
    * This is an enternal array used to copy data between buffers.
    */        
   protected byte[] array;
   
   /**
    * This is used to determine whether the consumer has finished.
    */ 
   protected boolean finished;
  
   /**
    * Constructor for the <code>LargeConsumer</code> object. This is
    * used to create a consumer with a one kilobyte buffer used to
    * read the contents from the cursor and transfer it to the buffer.
    */  
   protected LargeConsumer() {
      this(1024);
   }
   
   /**
    * Constructor for the <code>LargeConsumer</code> object. This is
    * used to create a consumer with a variable size buffer used to
    * read the contents from the cursor and transfer it to the buffer.
    *
    * @param chunk this is the size of the buffer used to read bytes
    */     
   protected LargeConsumer(int chunk) {
      this.array = new byte[chunk];
   }

   /**
    * This is used to determine whether the consumer has finished 
    * reading. The consumer is considered finished if it has read a
    * terminal token or if it has exhausted the stream and can not
    * read any more. Once finished the consumed bytes can be parsed.
    *
    * @return true if the consumer has finished reading its content
    */    
   public boolean isFinished() {
      return finished;
   }   
   
   /**
    * This method is used to consume bytes from the provided cursor.
    * Consuming of bytes from the cursor should be done in such a
    * way that it does not block. So typically only the number of
    * ready bytes in the <code>Cursor</code> object should be read.
    * If there are no ready bytes then this method should return.
    *
    * @param cursor used to consume the bytes from the HTTP pipeline
    */    
   public void consume(Cursor cursor) throws IOException {
      int ready = cursor.ready();      
      
      while(ready > 0) {
         int size = Math.min(ready, array.length);
         int count = cursor.read(array, 0, size);        

         if(count > 0) {
            int reset = process(array, 0, count);            
            
            if(reset > 0) {
               cursor.reset(reset);
            }
         }
         if(finished) {
        	   break;
         }
         ready = cursor.ready();        	 
      }
   } 
  
   /**
    * This is used to process the bytes that have been read from the
    * cursor. Depending on the delimeter used this knows when the
    * end of the body has been encountered. If the end is encountered
    * this method must return the number of bytes overflow, and set
    * the state of the consumer to finished.
    *
    * @param array this is a chunk read from the cursor
    * @param off this is the offset within the array the chunk starts
    * @param count this is the number of bytes within the array
    *
    * @return this returns the number of bytes overflow that is read
    */    
   protected abstract int process(byte[] array, int off, int count) throws IOException;
}


