/*
 * SecureNegotiator.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.transport;

import java.io.IOException;

import org.simpleframework.transport.reactor.ExecutorReactor;
import org.simpleframework.transport.reactor.Operation;
import org.simpleframework.transport.reactor.Reactor;
import org.simpleframework.util.thread.PoolExecutor;

/**
 * The <code>SecureNegotiator</code> object is used to negotiations
 * to complete SSL handshakes for secure connections. Negotiations 
 * are performed on <code>Pipeline</code> object before a transport
 * is created from the pipeline. Performing the SSL handshake is
 * required to ensure that only HTTP specific data is read and
 * written to the underlying transport.
 * 
 * @author Niall Gallagher
 */
class SecureNegotiator implements Negotiator {

   /**
    * This is the transport used to process complete transports.
    */
   private final Processor transporter;
   
   /**
    * This is the executor used to execute the negotiations.
    */
   private final PoolExecutor executor;
   
   /**
    * This is the reactor which is used to schedule I/O events.
    */
   private final Reactor reactor;

   /**
    * Constructor for the <code>SecureNegotiator</code> object. This
    * is used to create a negotiator that will perform SSL handshakes
    * on provided pipelines so that the data read from an written to
    * the underlying transport is complete and ready to use.
    * 
    * @param transporter this is used to process the transports
    * @param count this is the number of threads used by this
    */
   public SecureNegotiator(Processor transporter, int count) throws IOException {          
     this.executor = new PoolExecutor(count);          
     this.reactor = new ExecutorReactor(executor);            
     this.transporter = transporter;                  
   }  

   /**
    * This method is used to execute the provided operation without
    * the need to specifically check for I/O events. This is used if
    * the operation knows that the <code>SelectableChannel</code> is
    * ready, or if the I/O operation can be performed without knowing
    * if the channel is ready. Typically this is an efficient means
    * to perform a poll rather than a select on the channel.
    *
    * @param task this is the task to execute immediately
    */    
   public void process(Operation task) throws IOException {
     reactor.process(task);          
   }

   /**        
    * This method is used to execute the provided operation when there
    * is an I/O event that task is interested in. This will used the
    * operations <code>SelectableChannel</code> object to determine 
    * the events that are ready on the channel. If this reactor is
    * interested in any of the ready events then the task is executed.
    *
    * @param task this is the task to execute on interested events    
    * @param interest this is the bitmask value for interested events
    */   
   public void process(Operation task, int require) throws IOException {
     reactor.process(task, require);
   }

   /**
    * Once the negotiation has completed this is used to perform
    * processing of the provided transport. Processing of the
    * transport is done only after the negotiation has completed.
    * The given transport is used to read and write to the socket.
    * 
    * @param transport this is the transport for the pipeline
    */   
   public void process(Transport transport) throws IOException {
     transporter.process(transport); 
   }
   
   /**
    * This is used to stop the reactor so that further requests to
    * execute operations does nothing. This will clean up all of 
    * the reactors resources and unregister any operations that are
    * currently awaiting execution. This should be used to ensure
    * any threads used by the reactor gracefully stop.
    */    
   public void stop() throws IOException {
      reactor.stop();
      executor.stop();
   }
 }