/*
 * Reactor.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.transport.reactor;

import java.io.IOException;

/**
 * The <code>Reactor</code> interface is used to describe an object 
 * that is used to schedule asynchronous I/O operations. An operation
 * is performed by handing it to the reactor, which will determine
 * if an interested event has occurred. This allows the operation to
 * perform the task in a manner that does not block.
 * <p>
 * Implementing an <code>Operation</code> object requires that the
 * operation itself is aware of the I/O task it is performing. For
 * example, if the operation is concerned with reading data from the
 * underlying channel then the operation should perform the read, if
 * there is more data required then that operation to register with
 * the reactor again to receive further notifications. 
 *
 * @author Niall Gallagher
 *
 * @see org.simpleframework.transport.reactor.Operation
 */ 
public interface Reactor { 

  /**
   * This method is used to execute the provided operation without
   * the need to specifically check for I/O events. This is used if
   * the operation knows that the <code>SelectableChannel</code> is
   * ready, or if the I/O operation can be performed without knowing
   * if the channel is ready. Typically this is an efficient means
   * to perform a poll rather than a select on the channel.
   *
   * @param task this is the task to execute immediately
   */ 
  public void process(Operation task) throws IOException;

  /**        
   * This method is used to execute the provided operation when there
   * is an I/O event that task is interested in. This will used the
   * operations <code>SelectableChannel</code> object to determine 
   * the events that are ready on the channel. If this reactor is
   * interested in any of the ready events then the task is executed.
   *
   * @param task this is the task to execute on interested events    
   * @param require this is the bitmask value for interested events
   */
  public void process(Operation task, int require) throws IOException;

  /**
   * This is used to stop the reactor so that further requests to
   * execute operations does nothing. This will clean up all of 
   * the reactors resources and unregister any operations that are
   * currently awaiting execution. This should be used to ensure
   * any threads used by the reactor gracefully stop.
   */ 
  public void stop() throws IOException;
}




