/*
 * FilterAllocator.java February 2001
 *
 * Copyright (C) 2001, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.buffer;

import java.io.IOException;

/**
 * The <code>FilterAllocator</code> object is used to provide a means 
 * to provide a general set of constraints around buffer allocation. 
 * It can ensure that a minimum capacity is used for default allocation
 * and that an upper limit is used for allocation. In general this can
 * be used in conjunction with another <code>Allocator</code> which may
 * not have such constraints. It ensures that a set of requirements can
 * be observed when allocating buffers.
 * 
 * @author Niall Gallagher
 */ 
public class FilterAllocator implements Allocator {

   /**
    * This is the allocator the underlying buffer is allocated with.
    */         
   protected Allocator source;    

   /**
    * This is the default initial minimum capacity of the buffer.
    */ 
   protected int capacity;

   /**
    * This is the maximum number of bytes that can be allocated.
    */ 
   protected int limit;

   /**
    * Constructor for the <code>FilterAllocator</code> object. This is
    * used to instantiate the allocator with a default buffer size of
    * half a kilobyte. This ensures that it can be used for general 
    * purpose byte storage and for minor I/O tasks.
    *
    * @param source this is where the underlying buffer is allocated
    */   
   public FilterAllocator(Allocator source) {
      this(source, 512, 1048576);
   }

   /**
    * Constructor for the <code>FilterAllocator</code> object. This is
    * used to instantiate the allocator with a specified buffer size.
    * This is typically used when a very specific buffer capacity is
    * required, for example a request body with a known length.
    *
    * @param source this is where the underlying buffer is allocated    
    * @param capacity the initial capacity of the allocated buffers
    */
   public FilterAllocator(Allocator source, int capacity) {
      this(source, capacity, 1048576);
   }

   /**
    * Constructor for the <code>FilterAllocator</code> object. This is
    * used to instantiate the allocator with a specified buffer size.
    * This is typically used when a very specific buffer capacity is
    * required, for example a request body with a known length.
    *
    * @param source this is where the underlying buffer is allocated    
    * @param capacity the initial capacity of the allocated buffers
    * @param limit this is the maximum buffer size created by this
    */   
   public FilterAllocator(Allocator source, int capacity, int limit) {
      this.limit = Math.max(capacity, limit);
      this.capacity = capacity;
      this.source = source;
   }

   /**
    * This method is used to allocate a default buffer. This will 
    * allocate a buffer of predetermined size, allowing it to grow 
    * to an upper limit to accommodate extra data. If the buffer
    * requested is larger than the limit an exception is thrown.
    *
    * @return this returns an allocated buffer with a default size
    */    
   public Buffer allocate() throws IOException {     
      return allocate(capacity);
   }

   /**
    * This method is used to allocate a default buffer. This will 
    * allocate a buffer of predetermined size, allowing it to grow 
    * to an upper limit to accommodate extra data. If the buffer
    * requested is larger than the limit an exception is thrown.
    *
    * @param size the initial capacity of the allocated buffer
    *
    * @return this returns an allocated buffer with a default size
    */      
   public Buffer allocate(int size) throws IOException {
      if(size > limit) {
         throw new BufferException("Specified size %s beyond limit", size);
      }           
      if(capacity > size) {
         size = capacity;
      }    
      return source.allocate(size);
   }
   
   /**
    * This method is used to close the allocator so that resources
    * that are occupied by the allocator can be freed. This will
    * allow the allocator to be created and closed repeatedly in
    * a single process without holding on to resources such as
    * mapped file buffers or threads.
    */
   public void close() throws IOException {
      source.close();
   }
}