/*
 * Entry.java May 2004
 *
 * Copyright (C) 2004, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.lease;

import static java.util.concurrent.TimeUnit.NANOSECONDS;

import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;

/**
 * A <code>Entry</code> is used to represent the contract a lease
 * has been issued. This contains all relevant information for the
 * the lease, such as the keyed resource that has been leased and 
 * the duration of the lease. Durations for the contract can be 
 * measured in any <code>TimeUnit</code> for convenience.
 *
 * @author Niall Gallagher
 */
class Entry<T> implements Contract<T> {

   /**
    * This is the time that this entry expires in nanoseconds.
    */
   private volatile long time;
   
   /**
    * This is the key representing the resource being lease.
    */   
   private T key;
   
   /**
    * Constructor for the <code>Entry</code> object. This is used
    * to create a contract with an initial expiry period. Once this
    * is created the time is taken and the contract can be issued.    
    * 
    * @param key this is the key that this contract represents
    * @param lease this is the initial lease duration to be used
    * @param scale this is the time unit scale that is to be used
    */   
   public Entry(T key, long lease, TimeUnit scale) {
      this.time = getTime() + scale.toNanos(lease);
      this.key = key;
   }

   /**
    * This returns the key for the resource this represents. 
    * This is used when the contract has expired to clean resources
    * associated with the lease. It is passed in to the cleaner as
    * an parameter to the callback. The cleaner is then responsible
    * for cleaning any resources associated with the lease.    
    *
    * @return returns the resource key that this represents
    */   
   public T getKey() {
      return key;
   }

   /**
    * This method will return the number of <code>TimeUnit</code>
    * seconds that remain in the contract. If the value returned is 
    * less than or equal to zero then it should be assumed that the 
    * lease has expired, if greater than zero the lease is active.
    *
    * @return returns the duration in the time unit remaining
    */   
   public long getDelay(TimeUnit unit) {
      return unit.convert(time - getTime(), NANOSECONDS);
   }   

   /**
    * This method is used to set the number of <code>TimeUnit</code>
    * seconds that should remain within the contract. This is used
    * when the contract is to be reissued. Once a new duration has
    * been set the contract for the lease has been changed and the
    * previous expiry time is ignores, so only one clean is called.
    * 
    * @param delay this is the delay to be used for this contract
    * @param unit this is the time unit measurment for the delay
    */   
   public void setDelay(long delay, TimeUnit unit) {
      this.time = getTime() + unit.toNanos(delay);
   }

   /**
    * This method returns the current time in nanoseconds. This is
    * used to allow the duration of the lease to be calculated with
    * any given time unit which allows flexibility in setting and 
    * getting the current delay for the contract.
    *
    * @return returns the current time in nanoseconds remaining
    */   
   private long getTime() {
      return System.nanoTime();
   }
   
   /**
    * This is used to compare the specified delay to this delay. The
    * result of this operation is used to prioritize contracts in 
    * order of first to expire. Contracts that expire first reach
    * the top of the contract queue and are taken off for cleaning.
    * 
    * @param delay this is the delay to be compared with this
    * 
    * @return this returns zero if equal otherwise the difference
    */   
   public int compareTo(Delayed other) {
      Entry entry = (Entry) other;  
      
      if(other == this) {
         return 0;
      }
      return compareTo(entry);
   }
   
   /**
    * This is used to compare the specified delay to this delay. The
    * result of this operation is used to prioritize contracts in 
    * order of first to expire. Contracts that expire first reach
    * the top of the contract queue and are taken off for cleaning.
    * 
    * @param entry this is the entry to be compared with this
    * 
    * @return this returns zero if equal otherwise the difference
    */   
   private int compareTo(Entry entry) {
      long diff = time - entry.time;
      
      if(diff < 0) {
         return -1;
      } else if(diff > 0) {
         return 1;
      }
      return 0;
   }
   
   /**
    * This is used to provide a description of the contract that the
    * instance represents. A description well contain the key owned
    * by the contract as well as the expiry time expected for it.
    * This is used to provide descriptive messages in the exceptions.
    * 
    * @return a descriptive message describing the contract object
    */
   public String toString() {
      return String.format("contract %s", key);
   }
}
