/*
 * PoolExecutor.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.thread;

import java.util.concurrent.Executor;

/**
 * The <code>PoolExecutor</code> object is used to execute tasks in
 * a thread pool. This creates a thread pool with an unbounded list
 * of outstanding tasks, which ensures that any system requesting
 * a task to be executed will not block when handing it over.
 * 
 * @author Niall Gallagher
 */
public class PoolExecutor implements Executor {
   
   /**
    * This is the queue used to enqueue the tasks for execution.
    */
   private PoolQueue engine;
   
   /**
    * Constructor for the <code>PoolExecutor</code> object. This is
    * used to create a pool of threads that can be used to execute
    * arbitrary <code>Runnable</code> tasks. If the threads are
    * busy this will simply enqueue the tasks and return.
    */
   public PoolExecutor() {
      this(10);
   }
   
   /**
    * Constructor for the <code>PoolExecutor</code> object. This is
    * used to create a pool of threads that can be used to execute
    * arbitrary <code>Runnable</code> tasks. If the threads are
    * busy this will simply enqueue the tasks and return.
    * 
    * @param size this is the number of threads to use in the pool
    */   
   public PoolExecutor(int size) {
      this(size, size);
   }
   
   /**
    * Constructor for the <code>PoolExecutor</code> object. This is
    * used to create a pool of threads that can be used to execute
    * arbitrary <code>Runnable</code> tasks. If the threads are
    * busy this will simply enqueue the tasks and return.
    * 
    * @param rest this is the number of threads to use in the pool    
    * @param active this is the maximum size the pool can grow to 
    */   
   public PoolExecutor(int rest, int active) {     
      this.engine = new PoolQueue(rest, active);
   }   
   
   /**
    * The <code>execute</code> method is used to queue the task for
    * execution. If all threads are busy the provided task is queued
    * and waits until all current and outstanding tasks are finished.
    * 
    * @param task this is the task to be queued for execution
    */
   public void execute(Runnable task) {
      engine.execute(task);
   }
   
   /**
    * This is used to stop the executor by interrupting all running
    * tasks and shutting down the threads within the pool. This will
    * return immediately once it has been stopped, and not further
    * tasks will be accepted by this pool for execution.
    */
   public void stop() {
      engine.stop();
   }
}
