/*
 * PoolQueue.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.util.thread;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * The <code>PoolQueue</code> object is used to execute tasks in
 * a thread pool. This creates a thread pool with an unbounded list
 * of outstanding tasks, which ensures that any system requesting
 * a task to be executed will not block when handing it over.
 * 
 * @author Niall Gallagher
 */
class PoolQueue extends ThreadPoolExecutor {
   
   /**
    * Constructor for the <code>PoolQueue</code> object. This is
    * used to create a pool of threads that can be used to execute
    * arbitrary <code>Runnable</code> tasks. If the threads are
    * busy this will simply enqueue the tasks and return.
    * 
    * @param rest this is the number of threads to use in the pool    
    * @param active this is the maximum size the pool can grow to 
    */    
   public PoolQueue(int rest, int active) {
      this(rest, active, 120, TimeUnit.SECONDS);
   }
  
   /**
    * Constructor for the <code>PoolQueue</code> object. This is
    * used to create a pool of threads that can be used to execute
    * arbitrary <code>Runnable</code> tasks. If the threads are
    * busy this will simply enqueue the tasks and return.
    * 
    * @param rest this is the number of threads to use in the pool    
    * @param active this is the maximum size the pool can grow to
    * @param duration the duration active threads remain idle for
    * @param unit this is the time unit used for the duration 
    */    
   public PoolQueue(int rest, int active, long duration, TimeUnit unit) {
      super(rest, active, duration, unit, new LinkedBlockingQueue<Runnable>());
   }
   
   /**
    * This is used to stop the executor by interrupting all running
    * tasks and shutting down the threads within the pool. This will
    * return immediately once it has been stopped, and not further
    * tasks will be accepted by this pool for execution.
    */   
   public void stop() {
      shutdown();
   }
}