/*
 * PartConsumer.java February 2007
 *
 * Copyright (C) 2001, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;

import org.simpleframework.util.buffer.Allocator;

/**
 * The <code>PartConsumer</code> object is used to consume a part 
 * from a part list. A part consists of a header and a body, which
 * can be either a simple chunk of data or another part list. This
 * must be able to cope with either a simple body or a part list.
 *  
 * @author Niall Gallagher
 */ 
class PartConsumer implements Consumer {
 
   /**
    * This is used to consume the header message of the part.
    */  
   private SegmentConsumer header;
   
   /**
    * This is used to consume the body data from the part.
    */ 
   private BodyConsumer body;
   
   /**
    * This is used to determine what type the body data is.
    */ 
   private PartFactory factory;
   
   /**
    * This is the current consumer used to read from the cursor.
    */ 
   private Consumer current;
   
   /**
    * This is used to add the consumed parts to when finished.
    */  
   private PartList list;
   
   /**
    * This is the terminal token that ends the part payload.
    */ 
   private byte[] terminal;
   
   /**
    * Constructor for the <code>PartConsumer</code> object. This is
    * used to create a consumer used to read the contents of a part
    * and the boundary that terminates the content. Any parts that
    * are created by this are added to the provided part list.
    *
    * @param allocator this is the allocator used to creat buffers
    * @param list this is the part list used to store the parts
    * @param terminal this is the terminal token for the part
    */
   public PartConsumer(Allocator allocator, PartList list, byte[] terminal) {
      this.header = new PartHeaderConsumer(allocator);
      this.factory = new PartFactory(allocator, header);      
      this.terminal = terminal;
      this.current = header;
      this.list = list;
   }
 
   /**
    * This is used to create a new body consumer used to consume the
    * part body from for the list. This will ensure that the part 
    * data is created based on the part header consumed. The types 
    * of part supported are part lists and part body.
    *
    * @return this returns a consumed for the part content
    */   
   private BodyConsumer getConsumer() throws IOException {
      return factory.getInstance(list, terminal);
   }
   
   /** 
    * This is used to consume the part body from the cursor. This
    * initially reads the body of the part, which represents the
    * actual payload exposed via the <code>Part</code> interface
    * once the payload has been consumed the terminal is consumed.
    *
    * @param cursor this is the cursor to consume the body from
    */    
   public void consume(Cursor cursor) throws IOException {
      while(cursor.isReady()) {        
         if(header.isFinished()) {
            if(body == null) {
               body = getConsumer();
               current = body;
            } else {
               if(body.isFinished())             
                  break;               
            }
         }
         current.consume(cursor);
      }
   }
   
   /**
    * This is used to determine whether the part body has been read
    * from the cursor successfully. In order to determine if all of
    * the bytes have been read successfully this will check to see
    * of the terminal token had been consumed.
    *
    * @return true if the part body and terminal have been read 
    */    
   public boolean isFinished() {
      if(body != null) {
         return body.isFinished();
      }
      return false;
   }
}

