/*
 * PartEntryConsumer.java February 2007
 *
 * Copyright (C) 2007, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;

import org.simpleframework.util.buffer.Allocator;

/**
 * The <code>PartEntryConsumer</code> object is used to consume each
 * part from the part list. This is combines the task of consuming
 * the part, which consists of a header and a body, and a boundary
 * which identifies the end of the message content.
 *
 * @author Niall Gallagher
 */  
class PartEntryConsumer implements Consumer {
   
   /**
    * This is used to consume the boundary at the end of a part.
    */            
   private BoundaryConsumer boundary;

   /**
    * This is used to consume the actual part from the list.
    */  
   private Consumer consumer;
   
   /**
    * Constructor for the <code>PartEntryConsumer</code> object. This 
    * is used to create a consumer that will read the message part
    * and the boundary that terminates the part. All contents that
    * are read are appended to an internal buffer.
    *
    * @param allocator this is the allocator used for the buffer      
    * @param list this is the list used to accumulate the parts
    * @param terminal this is the terminal token for the part list 
    */       
   public PartEntryConsumer(Allocator allocator, PartList list, byte[] terminal) {
      this.consumer = new PartConsumer(allocator, list, terminal); 
      this.boundary = new BoundaryConsumer(allocator, terminal);        
   }
   
   /** 
    * This is used to consume the part body from the cursor. This
    * initially reads the body of the part, which represents the
    * actual content exposed via the <code>Part</code> interface
    * once the content has been consumed the terminal is consumed.
    *
    * @param cursor this is the cursor to consume the body from
    */       
   public void consume(Cursor cursor) throws IOException {
      while(cursor.isReady()) {
         if(!boundary.isFinished()) {
            boundary.consume(cursor);
         } else {               
            if(consumer.isFinished()) {
               break;
            }
            if(boundary.isEnd()) {
               break;
            }
            consumer.consume(cursor);
         }
      }
   }
   
   /**
    * This is used to determine whether the part body has been read
    * from the cursor successfully. In order to determine if all of
    * the bytes have been read successfully this will check to see
    * of the terminal token had been consumed.
    *
    * @return true if the part body and terminal have been read 
    */       
   public boolean isFinished() {
      if(boundary.isEnd()) {
         return true;
      }
      return consumer.isFinished();
   }

   /**
    * This is used to determine whether the terminal token read is
    * the final terminal token. The final terminal token is a 
    * normal terminal token, however it ends with two hyphens and
    * a carriage return line feed, this ends the part list.
    *
    * @return true if this was the last part within the list
    */      
   public boolean isEnd() {
      return boundary.isEnd();
   }
}