/*
 * PartFixedConsumer.java February 2008
 *
 * Copyright (C) 2008, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.http.core;

import java.io.IOException;
import java.io.InputStream;

import org.simpleframework.http.Part;
import org.simpleframework.util.buffer.Allocator;

/**
 * The <code>PartFixedConsumer</code> object is used to provide a
 * means to limit the number of bytes in a multipart list. This is
 * used to ensure that if the multipart body is not well formed 
 * that the part list consumer will not get caught in a loop where
 * it is constantly trying to consume the body. This also makes
 * the list consumer slightly more efficient as it does not need
 * to read and rest so many bytes.  
 * 
 * @author Niall Gallagher
 * 
 * @see org.simpleframework.http.core.PartListConsumer
 */
class PartFixedConsumer extends BodyConsumer {

   /**
    * This is the body consumer used to read the multipart list. 
    */
   private BodyConsumer consumer;
   
   /**
    * This is the remaining number of bytes withini the body.
    */
   private int remaining;   
   
   /**
    * This is the number of times the limit has been reached.
    */
   private int repeat;

   /**
    * Constructor for the <code>PartFixedConsumer</code> object. The
    * part fixed consumer is basically a wrapper that ensures that
    * no more thant the specified body length is consumed when the
    * multipart body is being consumed and parsed.
    * 
    * @param allocator this is the allocator used to store parts
    * @param boundary this is the boundary associated with a part
    * @param length this is the fixed number of bytes available
    */
   public PartFixedConsumer(Allocator allocator, byte[] boundary, int length) {
      this.consumer = new PartListConsumer(allocator, boundary, length);
      this.remaining = length;
   }

   /**
    * This method is used to acquire the limiting cursor. This can
    * be used by unit tests to ensure that the limiter is limiting
    * the number of bytes that can be read from the cursor.
    * 
    * @param cursor this is the cursor to wrap within a limiter
    * 
    * @return this will return a limiting cursor for the consumer
    */
   protected Cursor getCursor(Cursor cursor) {
      return new Limiter(cursor);
   }
   
   /** 
    * This is used to consume the part list from the cursor. This
    * initially reads the list of parts, which represents the
    * actual content exposed via the <code>PartList</code> object,
    * once the content has been consumed the terminal is consumed.
    *
    * @param cursor this is the cursor to consume the list from
    */ 
   public void consume(Cursor cursor) throws IOException {
      Cursor limit = getCursor(cursor);
     
      if(remaining > 0) {
         consumer.consume(limit);
      }
   }

   /**
    * This is used to determine whether the part body has been read
    * from the cursor successfully. In order to determine if all of
    * the bytes have been read successfully this will check to see
    * of the terminal token had been consumed.
    *
    * @return true if the part body and terminal have been read 
    */ 
   public boolean isFinished() {
      return consumer.isFinished();
   }

   /**
    * This is used to acquire the content of the part as a string.
    * The encoding of the string is taken from the content type. 
    * If no content type is sent the content is decoded in the
    * standard default of ISO-8859-1.
    * 
    * @return this returns a string representing the content
    */  
   @Override
   public String getContent() throws IOException {
      return consumer.getContent();
   }

   /**
    * This is used to acquire the content of the part as a string.
    * The encoding of the string is taken from the content type. 
    * If no content type is sent the content is decoded in the
    * standard default of ISO-8859-1.
    * 
    * @param charset this is the character encoding to be used
    *
    * @return this returns a string representing the content
    */  
   @Override
   public String getContent(String charset) throws IOException {
      return consumer.getContent(charset);
   }

   /**
    * This is used to acquire an <code>InputStream</code> for the
    * part. Acquiring the stream allows the content of the part to
    * be consumed by reading the stream. Each invocation of this
    * method will produce a new stream starting from the first byte.
    * 
    * @return this returns the stream for this part object
    */ 
   @Override
   public InputStream getInputStream() throws IOException {
      return consumer.getInputStream();
   }

   /**
    * This method is used to acquire a <code>Part</code> from the
    * body using a known name for the part. This is typically used 
    * when there is a file upload with a multipart POST request.
    * All parts that are not files are added to the query values
    * as strings so that they can be used in a convenient way.
    * 
    * @param name this is the name of the part to acquire
    * 
    * @return the named part or null if the part does not exist
    */ 
   @Override
   public Part getPart(String name) {
      return consumer.getPart(name);
   }

   /**
    * This method provides all parts for this body. The parts for a
    * body can contain text parameters or files. Each file part can
    * contain headers, which are the typical HTTP headers. Typically
    * headers describe the content and any encoding if required.
    * 
    * @return this returns a list of parts for this body
    */ 
   @Override
   public PartList getParts() {
      return consumer.getParts();
   }

   /**
    * The <code>Limiter</code> object is used to provide a wrapper
    * for a cursor that will limit the number of bytes read. This is
    * used to ensure that parsing errors do not result in excessive
    * bytes being consumed and also prevents looping.
    * 
    * @author Niall Gallagher
    */
   private class Limiter implements Cursor {

      /**
       * This is the cursor that this will delegate reading to.
       */
      private Cursor cursor;

      /**
       * Constructor for the <code>Limiter</code> object. This will
       * delegate reads to the provided cursor. Reads are limited
       * to the number of bytes remaining in the available length.
       * 
       * @param cursor this is the cursor to be delegated to
       */
      public Limiter(Cursor cursor) {
         this.cursor = cursor;
      }

      /**
       * Determines whether the cursor is still open. The cursor is
       * considered open if there are still bytes to read. If there 
       * is still bytes buffered and the underlying transport is 
       * closed then the cursor is still considered open. 
       * 
       * @return true if the read method returns a positive value
       */
      public boolean isOpen() throws IOException {
         return cursor.isOpen();
      }

      /**
       * Determines whether the cursor is ready for reading. When 
       * the cursor is ready then it guarantees that some amount of 
       * bytes can be read from the underlying stream.
       *
       * @return true if some data can be read without blocking
       */ 
      public boolean isReady() throws IOException {
         return ready() > 0;
      }

      /**
       * Reads a block of bytes from the underlying stream. This will
       * read up to the requested number of bytes from the underlying
       * stream. If there are no ready bytes on the stream this will
       * block, much like the <code>InputStream</code> a minus one is
       * returned if there are no more bytes to be read.
       *
       * @param data this is the array to read the bytes in to 
       *
       * @return returns the number of bytes read from the stream 
       */ 
      public int read(byte[] data) throws IOException {
         return read(data, 0, data.length);
      }

      /**
       * Reads a block of bytes from the underlying stream. This will
       * read up to the requested number of bytes from the underlying
       * stream. If there are no ready bytes on the stream this will
       * block, much like the <code>InputStream</code> a minus one is
       * returned if there are no more bytes to be read.
       *
       * @param data this is the array to read the bytes in to 
       * @param off this is the offset to begin writing the bytes to
       * @param len this is the number of bytes that are requested 
       *
       * @return returns the number of bytes read from the stream 
       */ 
      public int read(byte[] data, int off, int len) throws IOException {
         int size = Math.min(remaining, len);
         
         if(size <= 0) {
            return 0;
         }
         int count = cursor.read(data, off, size);

         if(count > 0) {
            remaining -= count;
         }
         if(remaining <= 0) {
            repeat++; // fully read
         }
         return count;
      }

      /**
       * Provides the number of bytes that can currently be read. The
       * number of bytes will be limited to the number of bytes that
       * remain within the available content length of the body.
       *
       * @return the number of bytes that can currently be read
       */ 
      public int ready() throws IOException {
         int ready = cursor.ready();

         if(ready > 0) {
            return Math.min(ready, remaining);
         }
         return 0;
      }

      /**
       * Moves the cursor backward within the stream. This ensures 
       * that any bytes read from the last read can be pushed back
       * in to the stream so that they can be read again. This will
       * throw an exception if the reset can not be performed.
       *
       * @param len this is the number of bytes to reset back
       *
       * @return this is the number of bytes that have been reset
       */
      public int reset(int size) throws IOException {
         int reset = cursor.reset(size);
         
         if(repeat > 100) {
            throw new IOException("Failed to read final boundary");
         }
         if(reset > 0) {
            remaining += reset;
         }
         return reset;
      }
   }
}
