/*
 * Writer.java February 2008
 *
 * Copyright (C) 2008, Niall Gallagher <niallg@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General 
 * Public License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA  02111-1307  USA
 */

package org.simpleframework.transport;

import java.io.IOException;
import java.nio.channels.SocketChannel;

/**
 * The <code>Writer</code> object represents a means to coalesce
 * packets at a single point before being written to the socket. It
 * is used to ensure all packets are queued in order of sequence
 * number. Any packets that are partially written to the underlying
 * socket can be coalesced in to a single packet so that a larger
 * packet can be delivered over the network.
 * 
 * @author Niall Gallagher
 * 
 * @see org.simpleframework.transport.Flusher
 */
interface Writer {

   /**
    * This is used to determine if the writer should block or not.
    * A writer will block only if there are shared packets still
    * within the write queue. When all shared packets have either
    * been written or duplicated then the writer does not need to
    * block any waiting threads and they can be released.
    * 
    * @return true if any writing thread should be blocked
    */
   public boolean isBlocking() throws IOException;
   
   /**
    * This is used to write the packets to the writer which will
    * be either written to the underlying socket or queued until
    * such time as the socket is write ready. This will return true
    * if the packet has been written to the underlying transport.
    * 
    * @param packet this is the packet that is the be written
    * 
    * @return true if the packet has been written to the transport
    */
   public boolean write(Packet packet) throws IOException;
   
   /**
    * This is used to flush all queued packets to the underlying
    * socket. If all of the queued packets have been fully written
    * then this returns true, otherwise this will return false.
    * 
    * @return true if all queued packets are flushed to the socket
    */
   public boolean flush() throws IOException;
   
   /**
    * This is used to close the writer and the underlying socket.
    * If a close is performed on the writer then no more bytes 
    * can be read from or written to the writer and the client 
    * will receive a connection close on their side.
    */  
   public void close() throws IOException;   
   
   /**
    * This provides the socket for the writer. Providing this 
    * enables a <code>Reactor</code> to be used to determine when
    * the writer is write ready and thus when the writer can
    * be flushed if it contains packets that have not been written.
    * 
    * @return this returns the socket associated with this
    */
   public SocketChannel getChannel();
}
