/*
 * Copyright 2018 SPF4J.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.spf4j.maven.plugin.avro.avscp.validation.impl;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.function.Consumer;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.avro.Schema;
import org.apache.avro.SchemaCompatibility;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.ArtifactResolutionException;
import org.eclipse.aether.resolution.VersionRangeResolutionException;
import org.eclipse.aether.version.Version;
import org.spf4j.io.compress.Compress;
import org.spf4j.maven.MavenRepositoryUtils;
import org.spf4j.maven.plugin.avro.avscp.SchemaCompileMojo;
import org.spf4j.maven.plugin.avro.avscp.ValidatorMojo;
import org.spf4j.maven.plugin.avro.avscp.validation.Validator;

/**
 * Validates all previously released schemas for backward compatibility.
 *
 * The following validations are performed:
 *
 * 1) We check previously released schema for (reader to writer) and (writer to reader) compatibility
 * with the current schema. Unless compatibility is not desired "beta", or only partial compatibility is desired
 * "noNewToOldCompatibility" "noOldToNewCompatibility"
 * (objects written with old schema don't need to be converted to new objects)
 * or "noNewToOldCompatibility" if (objects written with new schema don't need to be converted to old schema objects)
 *
 * Schema deprecation and removal policy check to be done at some point in the future.
 *
 * this validator can supports the following configuration keys:
 *
 * compatibiliy.versionRange - maven version range to check compatibility against. (defaults to
 * "[," + mavenProject.getVersion() +  ')' )
 *
 * compatibiliy.maxNrOfVersionsToCheckForCompatibility = max number oof versions to check against. (defaults to 30)
 *
 * compatibiliy.maxNrOfDaysBackCheckForCompatibility - max released time to check against (defaults to 1 year)
 *
 * @author Zoltan Farkas
 */
public final class SchemaCompatibilityValidator implements Validator<ValidatorMojo> {

  @Override
  public String getName() {
    return "compatibility";
  }

  @Override
  @SuppressFBWarnings("PRMC_POSSIBLY_REDUNDANT_METHOD_CALLS")
  public Result validate(final ValidatorMojo mojo) throws IOException {
    // loop through dependencies.
    MavenProject mavenProject = mojo.getMavenProject();
    Map<String, String> validatorConfigs = mojo.getValidatorConfigs();
    String versionRange = validatorConfigs.get("compatibiliy.versionRange");
    if (versionRange == null) {
      versionRange = "[," + mavenProject.getVersion() +  ')';
    }
    int maxNrVersToCheck = 30;
    String strNrVer = validatorConfigs.get("compatibiliy.maxNrOfVersionsToCheckForCompatibility");
    if (strNrVer != null) {
      maxNrVersToCheck = Integer.parseInt(strNrVer);
    }
    Instant instantToGoBack;
    String strNrDays = validatorConfigs.get("compatibiliy.maxNrOfDaysBackCheckForCompatibility");
    if (strNrDays == null) {
      instantToGoBack = Instant.now().atOffset(ZoneOffset.UTC).minus(1,  ChronoUnit.YEARS).toInstant();
    } else {
      instantToGoBack = Instant.now().atOffset(ZoneOffset.UTC)
              .minus(Integer.parseInt(strNrDays), ChronoUnit.DAYS).toInstant();
    }
    String groupId = mavenProject.getGroupId();
    String artifactId = mavenProject.getArtifactId();
    List<RemoteRepository> remoteProjectRepositories = mavenProject.getRemoteProjectRepositories();
    RepositorySystem repoSystem = mojo.getRepoSystem();
    RepositorySystemSession repositorySession = mojo.getMavenSession().getRepositorySession();
    List<Version> rangeVersions;
    try {
      rangeVersions = MavenRepositoryUtils.getVersions(groupId, artifactId, versionRange,
              remoteProjectRepositories, repoSystem, repositorySession);
    } catch (VersionRangeResolutionException ex) {
      throw new RuntimeException("Invalid compatibiliy.versionRange = " + versionRange + " setting", ex);
    }
    rangeVersions = rangeVersions.stream().filter((v) -> !v.toString().endsWith("SNAPSHOT"))
            .collect(Collectors.toList());
    mojo.getLog().info("Validating compatibility with previous versions " + rangeVersions);
    if (rangeVersions.isEmpty()) {
      return Result.valid();
    }
    List<String> issues = new ArrayList<>(4);
    int size = rangeVersions.size();
    for (int  i = size - 1, j = Math.max(size - maxNrVersToCheck, 0); i >= j; i--) {
      Version version  = rangeVersions.get(i);
      validateCompatibility(groupId, artifactId, version,
              remoteProjectRepositories, repoSystem, repositorySession, mojo, false, instantToGoBack, issues::add);
    }
    if (issues.isEmpty()) {
      return Result.valid();
    } else {
      return Result.failed("Schema compatibility issues:\n" + String.join("\n", issues));
    }
  }

  @SuppressFBWarnings("PCAIL_POSSIBLE_CONSTANT_ALLOCATION_IN_LOOP")
  public void validateCompatibility(final String groupId, final String artifactId, final Version version,
          final List<RemoteRepository> remoteProjectRepositories, final RepositorySystem repoSystem,
          final RepositorySystemSession repositorySession,
          final ValidatorMojo mojo,
          final boolean deprecationRemoval, final Instant instantToGoBack, final Consumer<String> issues)
          throws IOException {
    Log log = mojo.getLog();
    log.info("Validating compatibility with version: " + version);
    Path targetPath = mojo.getTarget().toPath();
    Path currSchemasPath = mojo.getGeneratedAvscTarget().toPath();
    File prevSchemaArchive;
    try {
      prevSchemaArchive = MavenRepositoryUtils.resolveArtifact(
              groupId, artifactId, null, "jar", version.toString(),
              remoteProjectRepositories, repoSystem, repositorySession);
    } catch (ArtifactResolutionException ex) {
      throw new RuntimeException("Cannot resolve previous version "  + version, ex);
    }
    Path dest = targetPath.resolve("prevSchemas").resolve(version.toString());
    Files.createDirectories(dest);
    List<Path> prevSchemas = Compress.unzip(prevSchemaArchive.toPath(), dest, (Path p) -> {
      Path fileName = p.getFileName();
      if (fileName == null) {
        return false;
      }
      String fname = fileName.toString();
      return (fname.endsWith("avsc")
              || SchemaCompileMojo.SCHEMA_MANIFEST.equals(fname)
              || "MANIFEST.MF".equals(fname));
    });
    Instant dependencyBuidTime = getDependencyBuidTime(dest, log);
    if (dependencyBuidTime.isBefore(instantToGoBack)) {
      return;
    }
    for (Path prevSchemaPath : prevSchemas) {
      Path fileName = prevSchemaPath.getFileName();
      if (fileName == null) {
        continue;
      }
      if (!fileName.toString().endsWith("avsc")) {
        continue;
      }
      Path relPath = dest.relativize(prevSchemaPath);
      Path newSchemaPath = currSchemasPath.resolve(relPath);
      Schema previousSchema = new Schema.Parser().parse(prevSchemaPath.toFile());
      String previousSchemaName = previousSchema.getFullName();
      if (previousSchema.getProp("beta") != null) {
        log.debug("Skipping beta schema " + previousSchemaName);
        continue;
      }
      log.debug("Validating compatibility for " + previousSchemaName + " "
              + prevSchemaPath + " -> "  + newSchemaPath);
      if (deprecationRemoval && !Files.exists(newSchemaPath) && previousSchema.getProp("deprecated") == null) {
        issues.accept(previousSchemaName + " is being removed without being deprecated first");
      } else {
        Schema newSchema = new Schema.Parser().parse(newSchemaPath.toFile());
        if (newSchema.getProp("beta") != null) {
          log.debug("Skipping beta schema " + newSchema.getFullName());
        } else {
          if (newSchema.getProp("noOldToNewCompatibility") == null) {
            SchemaCompatibility.SchemaPairCompatibility o2n
                    = SchemaCompatibility.checkReaderWriterCompatibility(newSchema, previousSchema);
            if (o2n.getType() == SchemaCompatibility.SchemaCompatibilityType.INCOMPATIBLE) {
              issues.accept(newSchema.getFullName() + " cannot convert previous versions " + version + " to current"
                    + " detail: " + o2n);
            }
          }
          if (newSchema.getProp("noNewToOldCompatibility") == null) {
            SchemaCompatibility.SchemaPairCompatibility n2o
                    = SchemaCompatibility.checkReaderWriterCompatibility(previousSchema, newSchema);
            if (n2o.getType() == SchemaCompatibility.SchemaCompatibilityType.INCOMPATIBLE) {
              issues.accept(newSchema.getFullName() + " cannot convert current to previos version " + version
              + " detail: " + n2o);
            }
          }
        }
      }
    }
  }

  @Nullable
  private static Instant getDependencyBuidTime(final Path location, final Log log) throws IOException {
    Path jarManifest = location.resolve("META-INF/MANIFEST.MF");
    if (Files.exists(jarManifest)) {
      try (BufferedInputStream bis = new BufferedInputStream(
              Files.newInputStream(jarManifest))) {
        Manifest manifest = new Manifest(bis);
        Attributes mainAttributes = manifest.getMainAttributes();
        String buildTime = mainAttributes.getValue("Build-Time");
        if (buildTime != null) {
          try {
            return Instant.parse(buildTime);
          } catch (DateTimeParseException ex) {
            log.warn("Cannot parse manifest build time " + buildTime, ex);
          }
        }
      }
    }
    Path codegenManifest = location.resolve(SchemaCompileMojo.SCHEMA_MANIFEST);
    if (Files.exists(codegenManifest)) {
      try (BufferedReader br = Files.newBufferedReader(codegenManifest, StandardCharsets.UTF_8)) {
        Properties props = new Properties();
        props.load(br);
        String buildTime = props.getProperty("Build-Time");
        if (buildTime != null) {
          try {
            return Instant.parse(buildTime);
          } catch (DateTimeParseException ex) {
            log.warn("Cannot parse manifest build time " + buildTime, ex);
          }
        }
      }
    }
    return null;
  }

  @Override
  public Class<ValidatorMojo> getValidationInput() {
    return ValidatorMojo.class;
  }

}
