/*
 * Decompiled with CFR 0.152.
 */
package org.visallo.common.rdf;

import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import com.hp.hpl.jena.datatypes.RDFDatatype;
import com.hp.hpl.jena.datatypes.xsd.XSDDatatype;
import com.hp.hpl.jena.datatypes.xsd.XSDDateTime;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.ResIterator;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.apache.commons.codec.digest.DigestUtils;
import org.json.JSONObject;
import org.vertexium.Authorizations;
import org.vertexium.Edge;
import org.vertexium.EdgeBuilder;
import org.vertexium.Element;
import org.vertexium.ElementBuilder;
import org.vertexium.Graph;
import org.vertexium.Metadata;
import org.vertexium.Property;
import org.vertexium.Vertex;
import org.vertexium.VertexBuilder;
import org.vertexium.Visibility;
import org.vertexium.mutation.ElementMutation;
import org.vertexium.property.StreamingPropertyValue;
import org.visallo.core.exception.VisalloException;
import org.visallo.core.ingest.graphProperty.GraphPropertyWorkData;
import org.visallo.core.model.ontology.OntologyRepository;
import org.visallo.core.model.properties.VisalloProperties;
import org.visallo.core.model.workQueue.Priority;
import org.visallo.core.model.workQueue.WorkQueueRepository;
import org.visallo.core.model.workspace.WorkspaceRepository;
import org.visallo.core.security.VisibilityTranslator;
import org.visallo.core.user.User;
import org.visallo.core.util.VisalloLogger;
import org.visallo.core.util.VisalloLoggerFactory;
import org.visallo.web.clientapi.model.VisibilityJson;

public class RdfXmlImportHelper {
    private static final VisalloLogger LOGGER = VisalloLoggerFactory.getLogger(RdfXmlImportHelper.class);
    public static final String RDF_TYPE_URI = "http://www.w3.org/1999/02/22-rdf-syntax-ns#type";
    private static final String MULTI_VALUE_KEY = RdfXmlImportHelper.class.getSimpleName();
    private final Graph graph;
    private final WorkQueueRepository workQueueRepository;
    private final String rdfConceptTypeIri;
    private final String hasEntityIri;
    private final WorkspaceRepository workspaceRepository;
    private final VisibilityTranslator visibilityTranslator;
    private boolean disableWorkQueues;

    public void setDisableWorkQueues(boolean disableWorkQueues) {
        this.disableWorkQueues = disableWorkQueues;
    }

    @Inject
    public RdfXmlImportHelper(Graph graph, WorkQueueRepository workQueueRepository, OntologyRepository ontologyRepository, WorkspaceRepository workspaceRepository, VisibilityTranslator visibilityTranslator) {
        this.graph = graph;
        this.workQueueRepository = workQueueRepository;
        this.workspaceRepository = workspaceRepository;
        this.visibilityTranslator = visibilityTranslator;
        this.hasEntityIri = ontologyRepository.getRequiredRelationshipIRIByIntent("artifactHasEntity");
        this.rdfConceptTypeIri = ontologyRepository.getConceptIRIByIntent("rdf");
    }

    public void importRdfXml(File inputFile, GraphPropertyWorkData data, Priority priority, String visibilitySource, User user, Authorizations authorizations) throws IOException {
        try (FileInputStream in = new FileInputStream(inputFile);){
            File baseDir = inputFile.getParentFile();
            this.importRdfXml(in, baseDir, data, priority, visibilitySource, user, authorizations);
        }
    }

    public void importRdfXml(InputStream in, File baseDir, GraphPropertyWorkData data, Priority priority, String visibilitySource, User user, Authorizations authorizations) {
        Visibility visibility = this.visibilityTranslator.toVisibility(visibilitySource).getVisibility();
        VisibilityJson visibilityJson = new VisibilityJson(visibilitySource);
        String workspaceId = null;
        if (data != null) {
            workspaceId = data.getWorkspaceId();
            visibilityJson.addWorkspace(workspaceId);
        }
        if (this.rdfConceptTypeIri != null && data != null) {
            VisalloProperties.CONCEPT_TYPE.setProperty(data.getElement(), (Object)this.rdfConceptTypeIri, this.visibilityTranslator.getDefaultVisibility(), authorizations);
        }
        Model model = ModelFactory.createDefaultModel();
        model.read(in, null);
        Results results = new Results();
        this.importRdfModel(results, model, baseDir, data, visibilityJson, visibility, user, authorizations);
        this.graph.flush();
        if (!this.disableWorkQueues) {
            LOGGER.debug("pushing vertices from RDF import on to work queue", new Object[0]);
            for (Vertex vertex : results.getVertices()) {
                this.workQueueRepository.broadcastElement((Element)vertex, workspaceId);
                for (Property prop : vertex.getProperties()) {
                    this.workQueueRepository.pushGraphPropertyQueue((Element)vertex, prop, priority);
                }
            }
            LOGGER.debug("pushing edges from RDF import on to work queue", new Object[0]);
            for (Edge edge : results.getEdges()) {
                this.workQueueRepository.broadcastElement((Element)edge, workspaceId);
                for (Property prop : edge.getProperties()) {
                    this.workQueueRepository.pushGraphPropertyQueue((Element)edge, prop, priority);
                }
            }
        }
    }

    private void importRdfModel(Results results, Model model, File baseDir, GraphPropertyWorkData data, VisibilityJson visibilityJson, Visibility visibility, User user, Authorizations authorizations) {
        ResIterator subjects = model.listSubjects();
        while (subjects.hasNext()) {
            Resource subject = (Resource)subjects.next();
            this.importSubject(results, this.graph, subject, baseDir, data, visibilityJson, visibility, user, authorizations);
        }
    }

    private void importSubject(Results results, Graph graph, Resource subject, File baseDir, GraphPropertyWorkData data, VisibilityJson visibilityJson, Visibility visibility, User user, Authorizations authorizations) {
        LOGGER.info("importSubject: %s", new Object[]{subject.toString()});
        String graphVertexId = this.getGraphVertexId(subject);
        VertexBuilder vertexBuilder = graph.prepareVertex(graphVertexId, visibility);
        if (data != null) {
            data.setVisibilityJsonOnElement((ElementBuilder)vertexBuilder);
        } else {
            VisalloProperties.VISIBILITY_JSON.setProperty((ElementMutation)vertexBuilder, (Object)visibilityJson, visibility);
        }
        VisalloProperties.MODIFIED_DATE.setProperty((ElementMutation)vertexBuilder, (Object)new Date(), visibility);
        VisalloProperties.MODIFIED_BY.setProperty((ElementMutation)vertexBuilder, (Object)user.getUserId(), visibility);
        StmtIterator statements = subject.listProperties();
        while (statements.hasNext()) {
            Statement statement = (Statement)statements.next();
            RDFNode obj = statement.getObject();
            if (obj instanceof Resource) {
                if (!this.isConceptTypeResource(statement)) continue;
                String value = statement.getResource().toString();
                VisalloProperties.CONCEPT_TYPE.setProperty((ElementMutation)vertexBuilder, (Object)value, this.visibilityTranslator.getDefaultVisibility());
                continue;
            }
            if (obj instanceof Literal) {
                LOGGER.info("set property on %s to %s", new Object[]{subject.toString(), statement.toString()});
                this.importLiteral(vertexBuilder, statement, baseDir, data, visibilityJson, visibility, user);
                continue;
            }
            throw new VisalloException("Unhandled object type: " + obj.getClass().getName());
        }
        Vertex v = vertexBuilder.save(authorizations);
        results.addVertex(v);
        if (data != null) {
            String edgeId = data.getElement().getId() + "_hasEntity_" + v.getId();
            EdgeBuilder e = graph.prepareEdge(edgeId, (Vertex)data.getElement(), v, this.hasEntityIri, visibility);
            data.setVisibilityJsonOnElement((ElementBuilder)e);
            results.addEdge(e.save(authorizations));
            this.addVertexToWorkspaceIfNeeded(data, v, user);
        }
        statements = subject.listProperties();
        while (statements.hasNext()) {
            Statement statement = (Statement)statements.next();
            RDFNode obj = statement.getObject();
            if (!(obj instanceof Resource) || this.isConceptTypeResource(statement)) continue;
            this.importResource(results, graph, v, statement, data, visibilityJson, visibility, user, authorizations);
        }
    }

    private boolean isConceptTypeResource(Statement statement) {
        String label = statement.getPredicate().toString();
        return label.equals(RDF_TYPE_URI);
    }

    private void importLiteral(VertexBuilder v, Statement statement, File baseDir, GraphPropertyWorkData data, VisibilityJson visibilityJson, Visibility visibility, User user) {
        Object literalValue;
        String propertyName = statement.getPredicate().toString();
        RDFDatatype datatype = statement.getLiteral().getDatatype();
        Object value = literalValue = statement.getLiteral().getValue();
        if (datatype == null || XSDDatatype.XSDstring.equals(datatype)) {
            String valueString = statement.getLiteral().toString();
            if (valueString.startsWith("streamingValue:")) {
                value = this.convertStreamingValueJsonToValueObject(baseDir, valueString);
            }
        } else if (literalValue instanceof XSDDateTime) {
            XSDDateTime xsdDateTime = (XSDDateTime)literalValue;
            value = xsdDateTime.asCalendar().getTime();
        } else {
            value = literalValue;
        }
        Metadata metadata = new Metadata();
        if (data != null) {
            metadata = data.createPropertyMetadata(user);
        } else {
            VisalloProperties.VISIBILITY_JSON_METADATA.setMetadata(metadata, (Object)visibilityJson, visibility);
            VisalloProperties.MODIFIED_BY_METADATA.setMetadata(metadata, (Object)user.getUserId(), visibility);
            VisalloProperties.MODIFIED_DATE_METADATA.setMetadata(metadata, (Object)new Date(), visibility);
        }
        v.addPropertyValue(MULTI_VALUE_KEY, propertyName, value, metadata, visibility);
    }

    private String hashValue(String valueString) {
        return DigestUtils.md5Hex((String)valueString).substring(0, 10);
    }

    private Object convertStreamingValueJsonToValueObject(File baseDir, String valueString) {
        FileInputStream in;
        JSONObject streamingValueJson = new JSONObject(valueString.substring("streamingValue:".length()));
        String fileName = streamingValueJson.getString("fileName");
        if (baseDir == null) {
            throw new VisalloException("Could not import streamingValue. No baseDir specified.");
        }
        File file = new File(baseDir, fileName);
        try {
            if (!file.exists()) {
                throw new VisalloException("File " + file.getAbsolutePath() + " does not exist.");
            }
            in = new FileInputStream(file);
        }
        catch (FileNotFoundException ex) {
            throw new VisalloException("File " + file.getAbsolutePath() + " does not exist.");
        }
        StreamingPropertyValue spv = new StreamingPropertyValue((InputStream)in, byte[].class);
        spv.searchIndex(false);
        spv.store(true);
        return spv;
    }

    private void importResource(Results results, Graph graph, Vertex outVertex, Statement statement, GraphPropertyWorkData data, VisibilityJson visibilityJson, Visibility visibility, User user, Authorizations authorizations) {
        String label = statement.getPredicate().toString();
        String vertexId = this.getGraphVertexId(statement.getResource());
        VertexBuilder inVertexBuilder = graph.prepareVertex(vertexId, visibility);
        if (data != null) {
            data.setVisibilityJsonOnElement((ElementBuilder)inVertexBuilder);
        }
        Vertex inVertex = inVertexBuilder.save(authorizations);
        results.addVertex(inVertex);
        if (data != null) {
            this.addVertexToWorkspaceIfNeeded(data, inVertex, user);
        }
        String edgeId = outVertex.getId() + "_" + label + "_" + inVertex.getId();
        EdgeBuilder e = graph.prepareEdge(edgeId, outVertex, inVertex, label, visibility);
        if (data != null) {
            data.setVisibilityJsonOnElement((ElementBuilder)e);
        } else {
            VisalloProperties.VISIBILITY_JSON.setProperty((ElementMutation)e, (Object)visibilityJson, visibility);
        }
        VisalloProperties.MODIFIED_BY.setProperty((ElementMutation)e, (Object)user.getUserId(), visibility);
        VisalloProperties.MODIFIED_DATE.setProperty((ElementMutation)e, (Object)new Date(), visibility);
        results.addEdge(e.save(authorizations));
        LOGGER.info("importResource: %s = %s", new Object[]{label, vertexId});
    }

    private String getGraphVertexId(Resource subject) {
        String subjectUri = subject.getURI();
        Preconditions.checkNotNull((Object)subjectUri, (Object)("could not get uri of subject: " + subject));
        int lastPound = subjectUri.lastIndexOf(35);
        Preconditions.checkArgument((lastPound >= 1 ? 1 : 0) != 0, (Object)("Could not find '#' in subject uri: " + subjectUri));
        return subjectUri.substring(lastPound + 1);
    }

    protected void addVertexToWorkspaceIfNeeded(GraphPropertyWorkData data, Vertex vertex, User user) {
        if (data.getWorkspaceId() == null) {
            return;
        }
        this.graph.flush();
        this.workspaceRepository.updateEntityOnWorkspace(data.getWorkspaceId(), vertex.getId(), Boolean.valueOf(false), null, user);
    }

    private static class Results {
        private final List<Vertex> vertices = new ArrayList<Vertex>();
        private final List<Edge> edges = new ArrayList<Edge>();

        private Results() {
        }

        public void addEdge(Edge edge) {
            this.edges.add(edge);
        }

        public void addVertex(Vertex vertex) {
            this.vertices.add(vertex);
        }

        public Iterable<Edge> getEdges() {
            return this.edges;
        }

        public Iterable<Vertex> getVertices() {
            return this.vertices;
        }
    }
}

