/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apprunner;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.apprunner.internal.AppRunnerServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.apprunner.model.AppRunnerException;
import software.amazon.awssdk.services.apprunner.model.AssociateCustomDomainRequest;
import software.amazon.awssdk.services.apprunner.model.AssociateCustomDomainResponse;
import software.amazon.awssdk.services.apprunner.model.CreateAutoScalingConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.CreateAutoScalingConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.CreateConnectionRequest;
import software.amazon.awssdk.services.apprunner.model.CreateConnectionResponse;
import software.amazon.awssdk.services.apprunner.model.CreateObservabilityConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.CreateObservabilityConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.CreateServiceRequest;
import software.amazon.awssdk.services.apprunner.model.CreateServiceResponse;
import software.amazon.awssdk.services.apprunner.model.CreateVpcConnectorRequest;
import software.amazon.awssdk.services.apprunner.model.CreateVpcConnectorResponse;
import software.amazon.awssdk.services.apprunner.model.CreateVpcIngressConnectionRequest;
import software.amazon.awssdk.services.apprunner.model.CreateVpcIngressConnectionResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteAutoScalingConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteAutoScalingConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteObservabilityConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteObservabilityConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteServiceRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteServiceResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteVpcConnectorRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteVpcConnectorResponse;
import software.amazon.awssdk.services.apprunner.model.DeleteVpcIngressConnectionRequest;
import software.amazon.awssdk.services.apprunner.model.DeleteVpcIngressConnectionResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeAutoScalingConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeAutoScalingConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeCustomDomainsRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeCustomDomainsResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeObservabilityConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeObservabilityConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeServiceRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeServiceResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeVpcConnectorRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeVpcConnectorResponse;
import software.amazon.awssdk.services.apprunner.model.DescribeVpcIngressConnectionRequest;
import software.amazon.awssdk.services.apprunner.model.DescribeVpcIngressConnectionResponse;
import software.amazon.awssdk.services.apprunner.model.DisassociateCustomDomainRequest;
import software.amazon.awssdk.services.apprunner.model.DisassociateCustomDomainResponse;
import software.amazon.awssdk.services.apprunner.model.InternalServiceErrorException;
import software.amazon.awssdk.services.apprunner.model.InvalidRequestException;
import software.amazon.awssdk.services.apprunner.model.InvalidStateException;
import software.amazon.awssdk.services.apprunner.model.ListAutoScalingConfigurationsRequest;
import software.amazon.awssdk.services.apprunner.model.ListAutoScalingConfigurationsResponse;
import software.amazon.awssdk.services.apprunner.model.ListConnectionsRequest;
import software.amazon.awssdk.services.apprunner.model.ListConnectionsResponse;
import software.amazon.awssdk.services.apprunner.model.ListObservabilityConfigurationsRequest;
import software.amazon.awssdk.services.apprunner.model.ListObservabilityConfigurationsResponse;
import software.amazon.awssdk.services.apprunner.model.ListOperationsRequest;
import software.amazon.awssdk.services.apprunner.model.ListOperationsResponse;
import software.amazon.awssdk.services.apprunner.model.ListServicesForAutoScalingConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.ListServicesForAutoScalingConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.ListServicesRequest;
import software.amazon.awssdk.services.apprunner.model.ListServicesResponse;
import software.amazon.awssdk.services.apprunner.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.apprunner.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.apprunner.model.ListVpcConnectorsRequest;
import software.amazon.awssdk.services.apprunner.model.ListVpcConnectorsResponse;
import software.amazon.awssdk.services.apprunner.model.ListVpcIngressConnectionsRequest;
import software.amazon.awssdk.services.apprunner.model.ListVpcIngressConnectionsResponse;
import software.amazon.awssdk.services.apprunner.model.PauseServiceRequest;
import software.amazon.awssdk.services.apprunner.model.PauseServiceResponse;
import software.amazon.awssdk.services.apprunner.model.ResourceNotFoundException;
import software.amazon.awssdk.services.apprunner.model.ResumeServiceRequest;
import software.amazon.awssdk.services.apprunner.model.ResumeServiceResponse;
import software.amazon.awssdk.services.apprunner.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.apprunner.model.StartDeploymentRequest;
import software.amazon.awssdk.services.apprunner.model.StartDeploymentResponse;
import software.amazon.awssdk.services.apprunner.model.TagResourceRequest;
import software.amazon.awssdk.services.apprunner.model.TagResourceResponse;
import software.amazon.awssdk.services.apprunner.model.UntagResourceRequest;
import software.amazon.awssdk.services.apprunner.model.UntagResourceResponse;
import software.amazon.awssdk.services.apprunner.model.UpdateDefaultAutoScalingConfigurationRequest;
import software.amazon.awssdk.services.apprunner.model.UpdateDefaultAutoScalingConfigurationResponse;
import software.amazon.awssdk.services.apprunner.model.UpdateServiceRequest;
import software.amazon.awssdk.services.apprunner.model.UpdateServiceResponse;
import software.amazon.awssdk.services.apprunner.model.UpdateVpcIngressConnectionRequest;
import software.amazon.awssdk.services.apprunner.model.UpdateVpcIngressConnectionResponse;
import software.amazon.awssdk.services.apprunner.transform.AssociateCustomDomainRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateAutoScalingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateConnectionRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateObservabilityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateVpcConnectorRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.CreateVpcIngressConnectionRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteAutoScalingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteConnectionRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteObservabilityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteVpcConnectorRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DeleteVpcIngressConnectionRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeAutoScalingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeCustomDomainsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeObservabilityConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeVpcConnectorRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DescribeVpcIngressConnectionRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.DisassociateCustomDomainRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListAutoScalingConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListConnectionsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListObservabilityConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListOperationsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListServicesForAutoScalingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListServicesRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListVpcConnectorsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ListVpcIngressConnectionsRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.PauseServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.ResumeServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.UpdateDefaultAutoScalingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.UpdateServiceRequestMarshaller;
import software.amazon.awssdk.services.apprunner.transform.UpdateVpcIngressConnectionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AppRunnerAsyncClient}.
 *
 * @see AppRunnerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppRunnerAsyncClient implements AppRunnerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAppRunnerAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAppRunnerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associate your own domain name with the App Runner subdomain URL of your App Runner service.
     * </p>
     * <p>
     * After you call <code>AssociateCustomDomain</code> and receive a successful response, use the information in the
     * <a>CustomDomain</a> record that's returned to add CNAME records to your Domain Name System (DNS). For each mapped
     * domain name, add a mapping to the target App Runner subdomain and one or more certificate validation records. App
     * Runner then performs DNS validation to verify that you own or control the domain name that you associated. App
     * Runner tracks domain validity in a certificate stored in <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide">AWS Certificate Manager (ACM)</a>.
     * </p>
     *
     * @param associateCustomDomainRequest
     * @return A Java Future containing the result of the AssociateCustomDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.AssociateCustomDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/AssociateCustomDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateCustomDomainResponse> associateCustomDomain(
            AssociateCustomDomainRequest associateCustomDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateCustomDomainRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateCustomDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateCustomDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateCustomDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateCustomDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateCustomDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateCustomDomainRequest, AssociateCustomDomainResponse>()
                            .withOperationName("AssociateCustomDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateCustomDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateCustomDomainRequest));
            CompletableFuture<AssociateCustomDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an App Runner automatic scaling configuration resource. App Runner requires this resource when you create
     * or update App Runner services and you require non-default auto scaling settings. You can share an auto scaling
     * configuration across multiple services.
     * </p>
     * <p>
     * Create multiple revisions of a configuration by calling this action multiple times using the same
     * <code>AutoScalingConfigurationName</code>. The call returns incremental
     * <code>AutoScalingConfigurationRevision</code> values. When you create a service and configure an auto scaling
     * configuration resource, the service uses the latest active revision of the auto scaling configuration by default.
     * You can optionally configure the service to use a specific revision.
     * </p>
     * <p>
     * Configure a higher <code>MinSize</code> to increase the spread of your App Runner service over more Availability
     * Zones in the Amazon Web Services Region. The tradeoff is a higher minimal cost.
     * </p>
     * <p>
     * Configure a lower <code>MaxSize</code> to control your cost. The tradeoff is lower responsiveness during peak
     * demand.
     * </p>
     *
     * @param createAutoScalingConfigurationRequest
     * @return A Java Future containing the result of the CreateAutoScalingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ServiceQuotaExceededException App Runner can't create this resource. You've reached your account
     *         quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.CreateAutoScalingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateAutoScalingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAutoScalingConfigurationResponse> createAutoScalingConfiguration(
            CreateAutoScalingConfigurationRequest createAutoScalingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAutoScalingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createAutoScalingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAutoScalingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAutoScalingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAutoScalingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAutoScalingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAutoScalingConfigurationRequest, CreateAutoScalingConfigurationResponse>()
                            .withOperationName("CreateAutoScalingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateAutoScalingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createAutoScalingConfigurationRequest));
            CompletableFuture<CreateAutoScalingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an App Runner connection resource. App Runner requires a connection resource when you create App Runner
     * services that access private repositories from certain third-party providers. You can share a connection across
     * multiple services.
     * </p>
     * <p>
     * A connection resource is needed to access GitHub and Bitbucket repositories. Both require a user interface
     * approval process through the App Runner console before you can use the connection.
     * </p>
     *
     * @param createConnectionRequest
     * @return A Java Future containing the result of the CreateConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ServiceQuotaExceededException App Runner can't create this resource. You've reached your account
     *         quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.CreateConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateConnection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConnectionResponse> createConnection(CreateConnectionRequest createConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConnectionRequest, CreateConnectionResponse>()
                            .withOperationName("CreateConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConnectionRequest));
            CompletableFuture<CreateConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an App Runner observability configuration resource. App Runner requires this resource when you create or
     * update App Runner services and you want to enable non-default observability features. You can share an
     * observability configuration across multiple services.
     * </p>
     * <p>
     * Create multiple revisions of a configuration by calling this action multiple times using the same
     * <code>ObservabilityConfigurationName</code>. The call returns incremental
     * <code>ObservabilityConfigurationRevision</code> values. When you create a service and configure an observability
     * configuration resource, the service uses the latest active revision of the observability configuration by
     * default. You can optionally configure the service to use a specific revision.
     * </p>
     * <p>
     * The observability configuration resource is designed to configure multiple features (currently one feature,
     * tracing). This action takes optional parameters that describe the configuration of these features (currently one
     * parameter, <code>TraceConfiguration</code>). If you don't specify a feature parameter, App Runner doesn't enable
     * the feature.
     * </p>
     *
     * @param createObservabilityConfigurationRequest
     * @return A Java Future containing the result of the CreateObservabilityConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ServiceQuotaExceededException App Runner can't create this resource. You've reached your account
     *         quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.CreateObservabilityConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateObservabilityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateObservabilityConfigurationResponse> createObservabilityConfiguration(
            CreateObservabilityConfigurationRequest createObservabilityConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createObservabilityConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createObservabilityConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateObservabilityConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateObservabilityConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateObservabilityConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateObservabilityConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateObservabilityConfigurationRequest, CreateObservabilityConfigurationResponse>()
                            .withOperationName("CreateObservabilityConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateObservabilityConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createObservabilityConfigurationRequest));
            CompletableFuture<CreateObservabilityConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an App Runner service. After the service is created, the action also automatically starts a deployment.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a href="https://docs.aws.amazon.com/apprunner/latest/api/API_ListOperations.html">ListOperations</a> call to
     * track the operation's progress.
     * </p>
     *
     * @param createServiceRequest
     * @return A Java Future containing the result of the CreateService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ServiceQuotaExceededException App Runner can't create this resource. You've reached your account
     *         quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.CreateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateServiceResponse> createService(CreateServiceRequest createServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateServiceRequest, CreateServiceResponse>()
                            .withOperationName("CreateService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createServiceRequest));
            CompletableFuture<CreateServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an App Runner VPC connector resource. App Runner requires this resource when you want to associate your
     * App Runner service to a custom Amazon Virtual Private Cloud (Amazon VPC).
     * </p>
     *
     * @param createVpcConnectorRequest
     * @return A Java Future containing the result of the CreateVpcConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ServiceQuotaExceededException App Runner can't create this resource. You've reached your account
     *         quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.CreateVpcConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateVpcConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVpcConnectorResponse> createVpcConnector(CreateVpcConnectorRequest createVpcConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVpcConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVpcConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVpcConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVpcConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVpcConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVpcConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVpcConnectorRequest, CreateVpcConnectorResponse>()
                            .withOperationName("CreateVpcConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVpcConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVpcConnectorRequest));
            CompletableFuture<CreateVpcConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Create an App Runner VPC Ingress Connection resource. App Runner requires this resource when you want to
     * associate your App Runner service with an Amazon VPC endpoint.
     * </p>
     *
     * @param createVpcIngressConnectionRequest
     * @return A Java Future containing the result of the CreateVpcIngressConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ServiceQuotaExceededException App Runner can't create this resource. You've reached your account
     *         quota for this resource type.</p>
     *         <p>
     *         For App Runner per-resource quotas, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/apprunner.html">App Runner endpoints and quotas</a>
     *         in the <i>Amazon Web Services General Reference</i>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.CreateVpcIngressConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/CreateVpcIngressConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVpcIngressConnectionResponse> createVpcIngressConnection(
            CreateVpcIngressConnectionRequest createVpcIngressConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVpcIngressConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVpcIngressConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVpcIngressConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVpcIngressConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVpcIngressConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVpcIngressConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVpcIngressConnectionRequest, CreateVpcIngressConnectionResponse>()
                            .withOperationName("CreateVpcIngressConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVpcIngressConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVpcIngressConnectionRequest));
            CompletableFuture<CreateVpcIngressConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an App Runner automatic scaling configuration resource. You can delete a top level auto scaling
     * configuration, a specific revision of one, or all revisions associated with the top level configuration. You
     * can't delete the default auto scaling configuration or a configuration that's used by one or more App Runner
     * services.
     * </p>
     *
     * @param deleteAutoScalingConfigurationRequest
     * @return A Java Future containing the result of the DeleteAutoScalingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DeleteAutoScalingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteAutoScalingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAutoScalingConfigurationResponse> deleteAutoScalingConfiguration(
            DeleteAutoScalingConfigurationRequest deleteAutoScalingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAutoScalingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteAutoScalingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAutoScalingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAutoScalingConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAutoScalingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAutoScalingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAutoScalingConfigurationRequest, DeleteAutoScalingConfigurationResponse>()
                            .withOperationName("DeleteAutoScalingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAutoScalingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAutoScalingConfigurationRequest));
            CompletableFuture<DeleteAutoScalingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an App Runner connection. You must first ensure that there are no running App Runner services that use
     * this connection. If there are any, the <code>DeleteConnection</code> action fails.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return A Java Future containing the result of the DeleteConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteConnection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConnectionResponse> deleteConnection(DeleteConnectionRequest deleteConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectionRequest, DeleteConnectionResponse>()
                            .withOperationName("DeleteConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConnectionRequest));
            CompletableFuture<DeleteConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an App Runner observability configuration resource. You can delete a specific revision or the latest
     * active revision. You can't delete a configuration that's used by one or more App Runner services.
     * </p>
     *
     * @param deleteObservabilityConfigurationRequest
     * @return A Java Future containing the result of the DeleteObservabilityConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DeleteObservabilityConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteObservabilityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteObservabilityConfigurationResponse> deleteObservabilityConfiguration(
            DeleteObservabilityConfigurationRequest deleteObservabilityConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteObservabilityConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteObservabilityConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteObservabilityConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteObservabilityConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteObservabilityConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteObservabilityConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteObservabilityConfigurationRequest, DeleteObservabilityConfigurationResponse>()
                            .withOperationName("DeleteObservabilityConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteObservabilityConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteObservabilityConfigurationRequest));
            CompletableFuture<DeleteObservabilityConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an App Runner service.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     * <note>
     * <p>
     * Make sure that you don't have any active VPCIngressConnections associated with the service you want to delete.
     * </p>
     * </note>
     *
     * @param deleteServiceRequest
     * @return A Java Future containing the result of the DeleteService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DeleteService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteServiceResponse> deleteService(DeleteServiceRequest deleteServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteServiceRequest, DeleteServiceResponse>()
                            .withOperationName("DeleteService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteServiceRequest));
            CompletableFuture<DeleteServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an App Runner VPC connector resource. You can't delete a connector that's used by one or more App Runner
     * services.
     * </p>
     *
     * @param deleteVpcConnectorRequest
     * @return A Java Future containing the result of the DeleteVpcConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DeleteVpcConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteVpcConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVpcConnectorResponse> deleteVpcConnector(DeleteVpcConnectorRequest deleteVpcConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVpcConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVpcConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVpcConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVpcConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVpcConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVpcConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVpcConnectorRequest, DeleteVpcConnectorResponse>()
                            .withOperationName("DeleteVpcConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVpcConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVpcConnectorRequest));
            CompletableFuture<DeleteVpcConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an App Runner VPC Ingress Connection resource that's associated with an App Runner service. The VPC
     * Ingress Connection must be in one of the following states to be deleted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED_CREATION</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED_UPDATE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED_DELETION</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteVpcIngressConnectionRequest
     * @return A Java Future containing the result of the DeleteVpcIngressConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DeleteVpcIngressConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DeleteVpcIngressConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVpcIngressConnectionResponse> deleteVpcIngressConnection(
            DeleteVpcIngressConnectionRequest deleteVpcIngressConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVpcIngressConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVpcIngressConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVpcIngressConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVpcIngressConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVpcIngressConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteVpcIngressConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVpcIngressConnectionRequest, DeleteVpcIngressConnectionResponse>()
                            .withOperationName("DeleteVpcIngressConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVpcIngressConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVpcIngressConnectionRequest));
            CompletableFuture<DeleteVpcIngressConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return a full description of an App Runner automatic scaling configuration resource.
     * </p>
     *
     * @param describeAutoScalingConfigurationRequest
     * @return A Java Future containing the result of the DescribeAutoScalingConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DescribeAutoScalingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeAutoScalingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAutoScalingConfigurationResponse> describeAutoScalingConfiguration(
            DescribeAutoScalingConfigurationRequest describeAutoScalingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeAutoScalingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAutoScalingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAutoScalingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAutoScalingConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeAutoScalingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAutoScalingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAutoScalingConfigurationRequest, DescribeAutoScalingConfigurationResponse>()
                            .withOperationName("DescribeAutoScalingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeAutoScalingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeAutoScalingConfigurationRequest));
            CompletableFuture<DescribeAutoScalingConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return a description of custom domain names that are associated with an App Runner service.
     * </p>
     *
     * @param describeCustomDomainsRequest
     * @return A Java Future containing the result of the DescribeCustomDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DescribeCustomDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeCustomDomains"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCustomDomainsResponse> describeCustomDomains(
            DescribeCustomDomainsRequest describeCustomDomainsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeCustomDomainsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCustomDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCustomDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCustomDomainsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeCustomDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCustomDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCustomDomainsRequest, DescribeCustomDomainsResponse>()
                            .withOperationName("DescribeCustomDomains").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeCustomDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeCustomDomainsRequest));
            CompletableFuture<DescribeCustomDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return a full description of an App Runner observability configuration resource.
     * </p>
     *
     * @param describeObservabilityConfigurationRequest
     * @return A Java Future containing the result of the DescribeObservabilityConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DescribeObservabilityConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeObservabilityConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeObservabilityConfigurationResponse> describeObservabilityConfiguration(
            DescribeObservabilityConfigurationRequest describeObservabilityConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeObservabilityConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeObservabilityConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeObservabilityConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeObservabilityConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeObservabilityConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeObservabilityConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeObservabilityConfigurationRequest, DescribeObservabilityConfigurationResponse>()
                            .withOperationName("DescribeObservabilityConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeObservabilityConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeObservabilityConfigurationRequest));
            CompletableFuture<DescribeObservabilityConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return a full description of an App Runner service.
     * </p>
     *
     * @param describeServiceRequest
     * @return A Java Future containing the result of the DescribeService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DescribeService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeServiceResponse> describeService(DescribeServiceRequest describeServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeServiceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeServiceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceRequest, DescribeServiceResponse>()
                            .withOperationName("DescribeService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeServiceRequest));
            CompletableFuture<DescribeServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return a description of an App Runner VPC connector resource.
     * </p>
     *
     * @param describeVpcConnectorRequest
     * @return A Java Future containing the result of the DescribeVpcConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DescribeVpcConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeVpcConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVpcConnectorResponse> describeVpcConnector(
            DescribeVpcConnectorRequest describeVpcConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVpcConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVpcConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVpcConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVpcConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVpcConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVpcConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVpcConnectorRequest, DescribeVpcConnectorResponse>()
                            .withOperationName("DescribeVpcConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeVpcConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeVpcConnectorRequest));
            CompletableFuture<DescribeVpcConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return a full description of an App Runner VPC Ingress Connection resource.
     * </p>
     *
     * @param describeVpcIngressConnectionRequest
     * @return A Java Future containing the result of the DescribeVpcIngressConnection operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DescribeVpcIngressConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DescribeVpcIngressConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeVpcIngressConnectionResponse> describeVpcIngressConnection(
            DescribeVpcIngressConnectionRequest describeVpcIngressConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVpcIngressConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVpcIngressConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVpcIngressConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeVpcIngressConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeVpcIngressConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeVpcIngressConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeVpcIngressConnectionRequest, DescribeVpcIngressConnectionResponse>()
                            .withOperationName("DescribeVpcIngressConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeVpcIngressConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeVpcIngressConnectionRequest));
            CompletableFuture<DescribeVpcIngressConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociate a custom domain name from an App Runner service.
     * </p>
     * <p>
     * Certificates tracking domain validity are associated with a custom domain and are stored in <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide">AWS Certificate Manager (ACM)</a>. These certificates
     * aren't deleted as part of this action. App Runner delays certificate deletion for 30 days after a domain is
     * disassociated from your service.
     * </p>
     *
     * @param disassociateCustomDomainRequest
     * @return A Java Future containing the result of the DisassociateCustomDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.DisassociateCustomDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/DisassociateCustomDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateCustomDomainResponse> disassociateCustomDomain(
            DisassociateCustomDomainRequest disassociateCustomDomainRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateCustomDomainRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateCustomDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateCustomDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateCustomDomainResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateCustomDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateCustomDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateCustomDomainRequest, DisassociateCustomDomainResponse>()
                            .withOperationName("DisassociateCustomDomain").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateCustomDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateCustomDomainRequest));
            CompletableFuture<DisassociateCustomDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of active App Runner automatic scaling configurations in your Amazon Web Services account. You can
     * query the revisions for a specific configuration name or the revisions for all active configurations in your
     * account. You can optionally query only the latest revision of each requested name.
     * </p>
     * <p>
     * To retrieve a full description of a particular configuration revision, call and provide one of the ARNs returned
     * by <code>ListAutoScalingConfigurations</code>.
     * </p>
     *
     * @param listAutoScalingConfigurationsRequest
     * @return A Java Future containing the result of the ListAutoScalingConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListAutoScalingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListAutoScalingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAutoScalingConfigurationsResponse> listAutoScalingConfigurations(
            ListAutoScalingConfigurationsRequest listAutoScalingConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAutoScalingConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAutoScalingConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAutoScalingConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAutoScalingConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAutoScalingConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAutoScalingConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAutoScalingConfigurationsRequest, ListAutoScalingConfigurationsResponse>()
                            .withOperationName("ListAutoScalingConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAutoScalingConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAutoScalingConfigurationsRequest));
            CompletableFuture<ListAutoScalingConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of App Runner connections that are associated with your Amazon Web Services account.
     * </p>
     *
     * @param listConnectionsRequest
     * @return A Java Future containing the result of the ListConnections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListConnections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListConnectionsResponse> listConnections(ListConnectionsRequest listConnectionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConnectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConnectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConnectionsRequest, ListConnectionsResponse>()
                            .withOperationName("ListConnections").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListConnectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConnectionsRequest));
            CompletableFuture<ListConnectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of active App Runner observability configurations in your Amazon Web Services account. You can
     * query the revisions for a specific configuration name or the revisions for all active configurations in your
     * account. You can optionally query only the latest revision of each requested name.
     * </p>
     * <p>
     * To retrieve a full description of a particular configuration revision, call and provide one of the ARNs returned
     * by <code>ListObservabilityConfigurations</code>.
     * </p>
     *
     * @param listObservabilityConfigurationsRequest
     * @return A Java Future containing the result of the ListObservabilityConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListObservabilityConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListObservabilityConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListObservabilityConfigurationsResponse> listObservabilityConfigurations(
            ListObservabilityConfigurationsRequest listObservabilityConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listObservabilityConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listObservabilityConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListObservabilityConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListObservabilityConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListObservabilityConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListObservabilityConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListObservabilityConfigurationsRequest, ListObservabilityConfigurationsResponse>()
                            .withOperationName("ListObservabilityConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListObservabilityConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listObservabilityConfigurationsRequest));
            CompletableFuture<ListObservabilityConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return a list of operations that occurred on an App Runner service.
     * </p>
     * <p>
     * The resulting list of <a>OperationSummary</a> objects is sorted in reverse chronological order. The first object
     * on the list represents the last started operation.
     * </p>
     *
     * @param listOperationsRequest
     * @return A Java Future containing the result of the ListOperations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListOperations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListOperationsResponse> listOperations(ListOperationsRequest listOperationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listOperationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOperations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOperationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOperationsRequest, ListOperationsResponse>()
                            .withOperationName("ListOperations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListOperationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listOperationsRequest));
            CompletableFuture<ListOperationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of running App Runner services in your Amazon Web Services account.
     * </p>
     *
     * @param listServicesRequest
     * @return A Java Future containing the result of the ListServices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListServices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListServices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicesResponse> listServices(ListServicesRequest listServicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listServicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListServicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServicesRequest, ListServicesResponse>()
                            .withOperationName("ListServices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServicesRequest));
            CompletableFuture<ListServicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the associated App Runner services using an auto scaling configuration.
     * </p>
     *
     * @param listServicesForAutoScalingConfigurationRequest
     * @return A Java Future containing the result of the ListServicesForAutoScalingConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListServicesForAutoScalingConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListServicesForAutoScalingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListServicesForAutoScalingConfigurationResponse> listServicesForAutoScalingConfiguration(
            ListServicesForAutoScalingConfigurationRequest listServicesForAutoScalingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listServicesForAutoScalingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listServicesForAutoScalingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListServicesForAutoScalingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListServicesForAutoScalingConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListServicesForAutoScalingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListServicesForAutoScalingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListServicesForAutoScalingConfigurationRequest, ListServicesForAutoScalingConfigurationResponse>()
                            .withOperationName("ListServicesForAutoScalingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListServicesForAutoScalingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listServicesForAutoScalingConfigurationRequest));
            CompletableFuture<ListServicesForAutoScalingConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List tags that are associated with for an App Runner resource. The response contains a list of tag key-value
     * pairs.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of App Runner VPC connectors in your Amazon Web Services account.
     * </p>
     *
     * @param listVpcConnectorsRequest
     * @return A Java Future containing the result of the ListVpcConnectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListVpcConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListVpcConnectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVpcConnectorsResponse> listVpcConnectors(ListVpcConnectorsRequest listVpcConnectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVpcConnectorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVpcConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVpcConnectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVpcConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVpcConnectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVpcConnectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVpcConnectorsRequest, ListVpcConnectorsResponse>()
                            .withOperationName("ListVpcConnectors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVpcConnectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVpcConnectorsRequest));
            CompletableFuture<ListVpcConnectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Return a list of App Runner VPC Ingress Connections in your Amazon Web Services account.
     * </p>
     *
     * @param listVpcIngressConnectionsRequest
     * @return A Java Future containing the result of the ListVpcIngressConnections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ListVpcIngressConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ListVpcIngressConnections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVpcIngressConnectionsResponse> listVpcIngressConnections(
            ListVpcIngressConnectionsRequest listVpcIngressConnectionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVpcIngressConnectionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVpcIngressConnectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVpcIngressConnections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVpcIngressConnectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVpcIngressConnectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListVpcIngressConnectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVpcIngressConnectionsRequest, ListVpcIngressConnectionsResponse>()
                            .withOperationName("ListVpcIngressConnections").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVpcIngressConnectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVpcIngressConnectionsRequest));
            CompletableFuture<ListVpcIngressConnectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Pause an active App Runner service. App Runner reduces compute capacity for the service to zero and loses state
     * (for example, ephemeral storage is removed).
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param pauseServiceRequest
     * @return A Java Future containing the result of the PauseService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.PauseService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/PauseService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PauseServiceResponse> pauseService(PauseServiceRequest pauseServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(pauseServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pauseServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PauseService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PauseServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PauseServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PauseServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PauseServiceRequest, PauseServiceResponse>()
                            .withOperationName("PauseService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PauseServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(pauseServiceRequest));
            CompletableFuture<PauseServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Resume an active App Runner service. App Runner provisions compute capacity for the service.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param resumeServiceRequest
     * @return A Java Future containing the result of the ResumeService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.ResumeService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/ResumeService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ResumeServiceResponse> resumeService(ResumeServiceRequest resumeServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resumeServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ResumeServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ResumeServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ResumeServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ResumeServiceRequest, ResumeServiceResponse>()
                            .withOperationName("ResumeService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ResumeServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(resumeServiceRequest));
            CompletableFuture<ResumeServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Initiate a manual deployment of the latest commit in a source code repository or the latest image in a source
     * image repository to an App Runner service.
     * </p>
     * <p>
     * For a source code repository, App Runner retrieves the commit and builds a Docker image. For a source image
     * repository, App Runner retrieves the latest Docker image. In both cases, App Runner then deploys the new image to
     * your service and starts a new container instance.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param startDeploymentRequest
     * @return A Java Future containing the result of the StartDeployment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.StartDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartDeploymentResponse> startDeployment(StartDeploymentRequest startDeploymentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDeploymentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeployment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDeploymentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDeploymentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                            .withOperationName("StartDeployment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startDeploymentRequest));
            CompletableFuture<StartDeploymentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add tags to, or update the tag values of, an App Runner resource. A tag is a key-value pair.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove tags from an App Runner resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an auto scaling configuration to be the default. The existing default auto scaling configuration will be
     * set to non-default automatically.
     * </p>
     *
     * @param updateDefaultAutoScalingConfigurationRequest
     * @return A Java Future containing the result of the UpdateDefaultAutoScalingConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.UpdateDefaultAutoScalingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/UpdateDefaultAutoScalingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDefaultAutoScalingConfigurationResponse> updateDefaultAutoScalingConfiguration(
            UpdateDefaultAutoScalingConfigurationRequest updateDefaultAutoScalingConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDefaultAutoScalingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDefaultAutoScalingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDefaultAutoScalingConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDefaultAutoScalingConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateDefaultAutoScalingConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDefaultAutoScalingConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDefaultAutoScalingConfigurationRequest, UpdateDefaultAutoScalingConfigurationResponse>()
                            .withOperationName("UpdateDefaultAutoScalingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDefaultAutoScalingConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDefaultAutoScalingConfigurationRequest));
            CompletableFuture<UpdateDefaultAutoScalingConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an App Runner service. You can update the source configuration and instance configuration of the service.
     * You can also update the ARN of the auto scaling configuration resource that's associated with the service.
     * However, you can't change the name or the encryption configuration of the service. These can be set only when you
     * create the service.
     * </p>
     * <p>
     * To update the tags applied to your service, use the separate actions <a>TagResource</a> and <a>UntagResource</a>.
     * </p>
     * <p>
     * This is an asynchronous operation. On a successful call, you can use the returned <code>OperationId</code> and
     * the <a>ListOperations</a> call to track the operation's progress.
     * </p>
     *
     * @param updateServiceRequest
     * @return A Java Future containing the result of the UpdateService operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.UpdateService
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/UpdateService" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceResponse> updateService(UpdateServiceRequest updateServiceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateService");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateServiceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceRequest, UpdateServiceResponse>()
                            .withOperationName("UpdateService").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateServiceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateServiceRequest));
            CompletableFuture<UpdateServiceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update an existing App Runner VPC Ingress Connection resource. The VPC Ingress Connection must be in one of the
     * following states to be updated:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AVAILABLE
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED_CREATION
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED_UPDATE
     * </p>
     * </li>
     * </ul>
     *
     * @param updateVpcIngressConnectionRequest
     * @return A Java Future containing the result of the UpdateVpcIngressConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException One or more input parameters aren't valid. Refer to the API action's document
     *         page, correct the input parameters, and try the action again.</li>
     *         <li>ResourceNotFoundException A resource doesn't exist for the specified Amazon Resource Name (ARN) in
     *         your Amazon Web Services account.</li>
     *         <li>InvalidStateException You can't perform this action when the resource is in its current state.</li>
     *         <li>InternalServiceErrorException An unexpected service exception occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AppRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AppRunnerAsyncClient.UpdateVpcIngressConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/apprunner-2020-05-15/UpdateVpcIngressConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVpcIngressConnectionResponse> updateVpcIngressConnection(
            UpdateVpcIngressConnectionRequest updateVpcIngressConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVpcIngressConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVpcIngressConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVpcIngressConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVpcIngressConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVpcIngressConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVpcIngressConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVpcIngressConnectionRequest, UpdateVpcIngressConnectionResponse>()
                            .withOperationName("UpdateVpcIngressConnection").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateVpcIngressConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateVpcIngressConnectionRequest));
            CompletableFuture<UpdateVpcIngressConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AppRunnerServiceClientConfiguration serviceClientConfiguration() {
        return new AppRunnerServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppRunnerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateException")
                                .exceptionBuilderSupplier(InvalidStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceErrorException")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        AppRunnerServiceClientConfigurationBuilder serviceConfigBuilder = new AppRunnerServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
