/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Amazon Simple Storage (Amazon S3) location and security configuration for <code>OfflineStore</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3StorageConfig implements SdkPojo, Serializable, ToCopyableBuilder<S3StorageConfig.Builder, S3StorageConfig> {
    private static final SdkField<String> S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S3Uri")
            .getter(getter(S3StorageConfig::s3Uri)).setter(setter(Builder::s3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Uri").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(S3StorageConfig::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> RESOLVED_OUTPUT_S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResolvedOutputS3Uri").getter(getter(S3StorageConfig::resolvedOutputS3Uri))
            .setter(setter(Builder::resolvedOutputS3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolvedOutputS3Uri").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_URI_FIELD,
            KMS_KEY_ID_FIELD, RESOLVED_OUTPUT_S3_URI_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3Uri;

    private final String kmsKeyId;

    private final String resolvedOutputS3Uri;

    private S3StorageConfig(BuilderImpl builder) {
        this.s3Uri = builder.s3Uri;
        this.kmsKeyId = builder.kmsKeyId;
        this.resolvedOutputS3Uri = builder.resolvedOutputS3Uri;
    }

    /**
     * <p>
     * The S3 URI, or location in Amazon S3, of <code>OfflineStore</code>.
     * </p>
     * <p>
     * S3 URIs have a format similar to the following: <code>s3://example-bucket/prefix/</code>.
     * </p>
     * 
     * @return The S3 URI, or location in Amazon S3, of <code>OfflineStore</code>.</p>
     *         <p>
     *         S3 URIs have a format similar to the following: <code>s3://example-bucket/prefix/</code>.
     */
    public final String s3Uri() {
        return s3Uri;
    }

    /**
     * <p>
     * The Amazon Web Services Key Management Service (KMS) key ARN of the key used to encrypt any objects written into
     * the <code>OfflineStore</code> S3 location.
     * </p>
     * <p>
     * The IAM <code>roleARN</code> that is passed as a parameter to <code>CreateFeatureGroup</code> must have below
     * permissions to the <code>KmsKeyId</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"kms:GenerateDataKey"</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Web Services Key Management Service (KMS) key ARN of the key used to encrypt any objects
     *         written into the <code>OfflineStore</code> S3 location.</p>
     *         <p>
     *         The IAM <code>roleARN</code> that is passed as a parameter to <code>CreateFeatureGroup</code> must have
     *         below permissions to the <code>KmsKeyId</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"kms:GenerateDataKey"</code>
     *         </p>
     *         </li>
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The S3 path where offline records are written.
     * </p>
     * 
     * @return The S3 path where offline records are written.
     */
    public final String resolvedOutputS3Uri() {
        return resolvedOutputS3Uri;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(resolvedOutputS3Uri());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3StorageConfig)) {
            return false;
        }
        S3StorageConfig other = (S3StorageConfig) obj;
        return Objects.equals(s3Uri(), other.s3Uri()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(resolvedOutputS3Uri(), other.resolvedOutputS3Uri());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3StorageConfig").add("S3Uri", s3Uri()).add("KmsKeyId", kmsKeyId())
                .add("ResolvedOutputS3Uri", resolvedOutputS3Uri()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3Uri":
            return Optional.ofNullable(clazz.cast(s3Uri()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "ResolvedOutputS3Uri":
            return Optional.ofNullable(clazz.cast(resolvedOutputS3Uri()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3StorageConfig, T> g) {
        return obj -> g.apply((S3StorageConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3StorageConfig> {
        /**
         * <p>
         * The S3 URI, or location in Amazon S3, of <code>OfflineStore</code>.
         * </p>
         * <p>
         * S3 URIs have a format similar to the following: <code>s3://example-bucket/prefix/</code>.
         * </p>
         * 
         * @param s3Uri
         *        The S3 URI, or location in Amazon S3, of <code>OfflineStore</code>.</p>
         *        <p>
         *        S3 URIs have a format similar to the following: <code>s3://example-bucket/prefix/</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Uri(String s3Uri);

        /**
         * <p>
         * The Amazon Web Services Key Management Service (KMS) key ARN of the key used to encrypt any objects written
         * into the <code>OfflineStore</code> S3 location.
         * </p>
         * <p>
         * The IAM <code>roleARN</code> that is passed as a parameter to <code>CreateFeatureGroup</code> must have below
         * permissions to the <code>KmsKeyId</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"kms:GenerateDataKey"</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyId
         *        The Amazon Web Services Key Management Service (KMS) key ARN of the key used to encrypt any objects
         *        written into the <code>OfflineStore</code> S3 location.</p>
         *        <p>
         *        The IAM <code>roleARN</code> that is passed as a parameter to <code>CreateFeatureGroup</code> must
         *        have below permissions to the <code>KmsKeyId</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"kms:GenerateDataKey"</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The S3 path where offline records are written.
         * </p>
         * 
         * @param resolvedOutputS3Uri
         *        The S3 path where offline records are written.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedOutputS3Uri(String resolvedOutputS3Uri);
    }

    static final class BuilderImpl implements Builder {
        private String s3Uri;

        private String kmsKeyId;

        private String resolvedOutputS3Uri;

        private BuilderImpl() {
        }

        private BuilderImpl(S3StorageConfig model) {
            s3Uri(model.s3Uri);
            kmsKeyId(model.kmsKeyId);
            resolvedOutputS3Uri(model.resolvedOutputS3Uri);
        }

        public final String getS3Uri() {
            return s3Uri;
        }

        public final void setS3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
        }

        @Override
        public final Builder s3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getResolvedOutputS3Uri() {
            return resolvedOutputS3Uri;
        }

        public final void setResolvedOutputS3Uri(String resolvedOutputS3Uri) {
            this.resolvedOutputS3Uri = resolvedOutputS3Uri;
        }

        @Override
        public final Builder resolvedOutputS3Uri(String resolvedOutputS3Uri) {
            this.resolvedOutputS3Uri = resolvedOutputS3Uri;
            return this;
        }

        @Override
        public S3StorageConfig build() {
            return new S3StorageConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
