/*
 * Copyright (c) 2020. zhangbohan.dell@gmail.com ,All Rights Reserved
 *
 */

package xyz.zhangbohan.common.tools;

import com.google.common.io.Files;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.binary.Base64;

import javax.crypto.Cipher;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.HashMap;
import java.util.Map;
/**
 * file encoding: utf-8
 * Function :
 * Create : 9/27/2019 4:08 PM
 *
 * @author : zhangbohan.dell@gmail.com
 * @version : 1.0
 */

@Slf4j
public class EncryptUtils {



	/**
	 * 非对称加密算法RSA算法组件
	 * 非对称算法一般是用来传送对称加密算法的密钥来使用的，相对于DH算法，RSA算法只需要一方构造密钥，不需要
	 * 大费周章的构造各自本地的密钥对了。DH算法只能算法非对称算法的底层实现。而RSA算法算法实现起来较为简单
	 *
	 * @author kongqz
	 */
	//非对称密钥算法
	public static final String KEY_ALGORITHM = "RSA";


	/**
	 * 密钥长度，DH算法的默认密钥长度是1024
	 * 密钥长度必须是64的倍数，在512到65536位之间
	 */
	private static final int KEY_SIZE = 2048;
	//公钥
	private static final String PUBLIC_KEY = "RSAPublicKey";

	//私钥
	private static final String PRIVATE_KEY = "RSAPrivateKey";

	/**
	 * 初始化密钥对
	 *
	 * @return Map 甲方密钥的Map
	 */
	public static Map<String, Object> initKey() throws Exception {
		//实例化密钥生成器
		KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(KEY_ALGORITHM);
		//初始化密钥生成器
		keyPairGenerator.initialize(KEY_SIZE);
		//生成密钥对
		KeyPair keyPair = keyPairGenerator.generateKeyPair();
		//甲方公钥
		RSAPublicKey publicKey = (RSAPublicKey) keyPair.getPublic();
		//甲方私钥
		RSAPrivateKey privateKey = (RSAPrivateKey) keyPair.getPrivate();
		//将密钥存储在map中
		Map<String, Object> keyMap = new HashMap<String, Object>();
		keyMap.put(PUBLIC_KEY, publicKey);
		keyMap.put(PRIVATE_KEY, privateKey);
		return keyMap;
	}


	/**
	 * 私钥加密
	 *
	 * @param data 待加密数据
	 * @param key  密钥
	 * @return byte[] 加密数据
	 */
	public static byte[] encryptByPrivateKey(byte[] data, byte[] key) throws Exception {

		//取得私钥
		PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(key);
		KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
		//生成私钥
		PrivateKey privateKey = keyFactory.generatePrivate(pkcs8KeySpec);
		//数据加密
		Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
		cipher.init(Cipher.ENCRYPT_MODE, privateKey);

		int blockSize = 245;
		int outputSize = cipher.getOutputSize(data.length);
		int leavedSize = data.length % blockSize;
		int blocksSize = leavedSize != 0 ? data.length / blockSize + 1 : data.length / blockSize;

		byte[] raw = new byte[outputSize * blocksSize];
		int i =0;
		while (data.length - i * blockSize > 0) {
			if (data.length - i * blockSize > blockSize) {
				cipher.doFinal(data, i * blockSize, blockSize, raw, i * outputSize);
			}else {
				cipher.doFinal(data, i * blockSize, data.length-i*blockSize, raw, i * outputSize);
			}
			i++;
		}
		return raw;
	}

	/**
	 * 公钥加密
	 *
	 * @param data 待加密数据
	 * @param key  密钥
	 * @return byte[] 加密数据
	 */
	public static byte[] encryptByPublicKey(byte[] data, byte[] key) throws Exception {

		//实例化密钥工厂
		KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
		//初始化公钥
		//密钥材料转换
		X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(key);
		//产生公钥
		PublicKey pubKey = keyFactory.generatePublic(x509KeySpec);

		//数据加密
		Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
		cipher.init(Cipher.ENCRYPT_MODE, pubKey);

		int blockSize = 245;
		int outputSize = cipher.getOutputSize(data.length);
		int leavedSize = data.length % blockSize;
		int blocksSize = leavedSize != 0 ? data.length / blockSize + 1 : data.length / blockSize;

		byte[] raw = new byte[outputSize * blocksSize];
		int i =0;
		while (data.length - i * blockSize > 0) {
			if (data.length - i * blockSize > blockSize) {
				cipher.doFinal(data, i * blockSize, blockSize, raw, i * outputSize);
			}else {
				cipher.doFinal(data, i * blockSize, data.length-i*blockSize, raw, i * outputSize);
			}
			i++;
		}
		return raw;



	}

	/**
	 * 私钥解密
	 *
	 * @param data 待解密数据
	 * @param key  密钥
	 * @return byte[] 解密数据
	 */
	public static byte[] decryptByPrivateKey(byte[] data, byte[] key) throws Exception {
		//取得私钥
		PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(key);
		KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
		//生成私钥
		PrivateKey privateKey = keyFactory.generatePrivate(pkcs8KeySpec);
		//数据解密
		Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
		cipher.init(Cipher.DECRYPT_MODE, privateKey);
		int blockSize = 256;
		ByteArrayOutputStream arrayOutputStream = new ByteArrayOutputStream(1024);
		int j = 0;
		while (data.length - j * blockSize > 0) {
			arrayOutputStream.write(cipher.doFinal(data, j * blockSize, blockSize));
			j++;
		}
		return arrayOutputStream.toByteArray();
	}

	/**
	 * 公钥解密
	 *
	 * @param data 待解密数据
	 * @param key  密钥
	 * @return byte[] 解密数据
	 */
	public static byte[] decryptByPublicKey(byte[] data, byte[] key) throws Exception {

		//实例化密钥工厂
		KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
		//初始化公钥
		//密钥材料转换
		X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(key);
		//产生公钥
		PublicKey pubKey = keyFactory.generatePublic(x509KeySpec);
		//数据解密
		Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
		cipher.init(Cipher.DECRYPT_MODE, pubKey);
		int blockSize = 256;
		ByteArrayOutputStream arrayOutputStream = new ByteArrayOutputStream(1024);
		int j = 0;
		while (data.length - j * blockSize > 0) {
			arrayOutputStream.write(cipher.doFinal(data, j * blockSize, blockSize));
			j++;
		}
		return arrayOutputStream.toByteArray();
	}

	/**
	 * 取得私钥
	 *
	 * @param keyMap 密钥map
	 * @return byte[] 私钥
	 */
	public static byte[] getPrivateKey(Map<String, Object> keyMap) {
		Key key = (Key) keyMap.get(PRIVATE_KEY);
		return key.getEncoded();
	}

	/**
	 * 取得公钥
	 *
	 * @param keyMap 密钥map
	 * @return byte[] 公钥
	 */
	public static byte[] getPublicKey(Map<String, Object> keyMap) throws Exception {
		Key key = (Key) keyMap.get(PUBLIC_KEY);
		return key.getEncoded();
	}

	/**
	 * @throws Exception
	 */
	public static void testInitlize() throws Exception {
		//初始化密钥
		//生成密钥对
		Map<String, Object> keyMap = EncryptUtils.initKey();
		//公钥
		byte[] publicKey = EncryptUtils.getPublicKey(keyMap);

		//私钥
		byte[] privateKey = EncryptUtils.getPrivateKey(keyMap);
		System.out.println("公钥：/n" + Base64.encodeBase64String(publicKey));
		System.out.println("私钥：/n" + Base64.encodeBase64String(privateKey));

		System.out.println("================密钥对构造完毕,甲方将公钥公布给乙方，开始进行加密数据的传输=============");
		String str = "RSA密码交换算法";
		System.out.println("/n===========甲方向乙方发送加密数据==============");
		System.out.println("原文:" + str);
		//甲方进行数据的加密
		byte[] code1 = EncryptUtils.encryptByPrivateKey(str.getBytes(), privateKey);
		System.out.println("加密后的数据：" + Base64.encodeBase64String(code1));
		System.out.println("===========乙方使用甲方提供的公钥对数据进行解密==============");
		//乙方进行数据的解密
		byte[] decode1 = EncryptUtils.decryptByPublicKey(code1, publicKey);
		System.out.println("乙方解密后的数据：" + new String(decode1) + "/n/n");

		System.out.println("===========反向进行操作，乙方向甲方发送数据==============/n/n");

		str = "乙方向甲方发送数据RSA算法";

		System.out.println("原文:" + str);

		//乙方使用公钥对数据进行加密
		byte[] code2 = EncryptUtils.encryptByPublicKey(str.getBytes(), publicKey);
		System.out.println("===========乙方使用公钥对数据进行加密==============");
		System.out.println("加密后的数据：" + Base64.encodeBase64String(code2));

		System.out.println("=============乙方将数据传送给甲方======================");
		System.out.println("===========甲方使用私钥对数据进行解密==============");

		//甲方使用私钥对数据进行解密
		byte[] decode2 = EncryptUtils.decryptByPrivateKey(code2, privateKey);

		System.out.println("甲方解密后的数据：" + new String(decode2));
	}


	/**
	 * 模拟发送方 生成秘钥，并保存秘钥
	 * (通过其他方式把秘钥提供给接收方，邮件，网络，U盘...)
	 */
	public static void saveSecret(String publicPath,String privatePath) throws Exception {
		Map<String, Object> keyMap = initKey();
		//公钥
		byte[] publicKey = EncryptUtils.getPublicKey(keyMap);

		//私钥
		byte[] privateKey = EncryptUtils.getPrivateKey(keyMap);

		File publicKeyFile = new File(publicPath);
		File privateKeyFile = new File(privatePath);
		if (!publicKeyFile.exists()) {
			privateKeyFile.createNewFile();
		}
		if (!privateKeyFile.exists()) {
			privateKeyFile.createNewFile();
		}


		Files.write(publicKey,publicKeyFile);
		Files.write(privateKey,privateKeyFile);


	}




}
