package ai.digital.config

import com.xebialabs.deployit.util.PasswordUtil.{APPLICATION_PASSWORDS, KEY_REPOSITORY_KEYSTORE_PASSWORD}
import com.xebialabs.deployit.{ServerConfigFile, ServerConfiguration}
import com.xebialabs.deployit.util.{DeployitKeys, PasswordEncrypter, PropertyUtil}

import java.io.File

import scala.jdk.CollectionConverters.CollectionHasAsScala

object ServerConfigurationHelper {

  val ConfigAdminUser: String = "_xl-deploy-config-admin_"
  val ConfigRole: String = "CONFIG_ADMIN"
  val DEFAULT_CONFIGURATION_FILE = new File("conf", "deployit.conf")

  private val defaultEncrypter = new PasswordEncrypter(DeployitKeys.DEFAULT_PASSWORD_ENCRYPTION_KEY)
  private val HTTP_PORT: Int = 8888

  def init(): ServerConfiguration = {
    val configFile = new ServerConfigFile(DEFAULT_CONFIGURATION_FILE)
    val config: ServerConfiguration =
      if (configFile.exists()) {
        configFile.loadConfig(false, false, true)
      } else {
        val serverConfiguration = new ServerConfiguration(true)
        serverConfiguration.loadDefault()
        serverConfiguration.setHttpPort(HTTP_PORT)
        serverConfiguration
      }

    val passwordEncryptionKey = DeployitKeys.getPasswordEncryptionKey(config.getRepositoryKeyStorePassword)
    PasswordEncrypter.init(passwordEncryptionKey)
    encryptPasswords()
    ServerConfiguration.setInstance(config)
    config
  }

  private def encryptPasswords(): Unit = {
    if(DEFAULT_CONFIGURATION_FILE.exists()) {
      val properties = PropertyUtil.readPropertiesFile(DEFAULT_CONFIGURATION_FILE)
      properties.stringPropertyNames.asScala
        .filter(key => APPLICATION_PASSWORDS.contains(key))
        .foreach(key => {
          if (KEY_REPOSITORY_KEYSTORE_PASSWORD.equals(key)) {
            properties.setProperty(key,
              defaultEncrypter.ensureEncrypted(properties.getProperty(key)))
          } else {
            properties.setProperty(key,
              Option(properties.getProperty(key)).map(PasswordEncrypter.getInstance().ensureEncrypted).get)
          }
        })
      PropertyUtil.storePropertiesFile(DEFAULT_CONFIGURATION_FILE, properties)
    }
  }

  def getAdminPassword: Option[String] =
    if (ServerConfiguration.getInstance().getSpringCloudExternalConfig) {
      // if we provide an option to change the admin password for external cc, then we have to add
      // a corresponding key in the clients to store that password and send it as part of the HTTP
      // requests through "spring.cloud.config.password"
      Option(ServerConfiguration.DEFAULT_ADMIN_PASSWORD)
    } else {
      Option(ServerConfiguration.getInstance().getAdminPassword)
        .map(PasswordEncrypter.getInstance().ensureDecrypted)
    }
}
