package ai.digital.deploy.cache.config

import ai.digital.configuration.central.deploy.CachesProperties
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.util.StringUtils

import java.net.URI
import javax.cache.spi.CachingProvider
import javax.cache.{CacheManager, Caching}
import scala.jdk.CollectionConverters._

@Configuration
class CacheConfiguration(@Autowired cachesProperties: CachesProperties,
                        @Autowired cacheConfigurationValidator: CacheConfigurationValidator)
  extends Logging {

  @Bean(destroyMethod = "close")
  def cacheProvider: CacheProvider = {
    try {
      if (!cachesProperties.enabled) {
        logger.error("Caching not enabled in properties. CacheManager not created.")
        return null
      }
      if (!cacheConfigurationValidator.validate) {
        logger.error("Cache configuration validation failed. CacheManager not created.")
        return null
      }
      new CacheProvider(getCacheManager)
    } catch {
      case exception: Exception =>
        logger.error(s"Exception occurred while creating CacheManager, $exception")
        null
    }
  }

  private def getCacheManager: CacheManager = {
    val cacheManager = if (cachesProperties.providerConfiguration.embedded) {
      getEmbeddedCacheManager
    } else {
      getStandAloneCacheManager
    }
    logger.info(s"cacheManager($cacheManager) of provider(${cacheManager.getCachingProvider}) created.")
    cacheManager
  }

  private def getEmbeddedCacheManager: CacheManager = {
    Caching.getCachingProvider(DEFAULT_EMBEDDED_CACHING_PROVIDER).getCacheManager
  }

  private def getStandAloneCacheManager: CacheManager = {
    getProvider.getCacheManager(getProviderConfigurationFileURI, null)
  }

  private def getProviderConfigurationFileURI: URI = {
    getResource(cachesProperties.providerConfiguration.providerConfigurationFile).toURI
  }

  private def getProvider: CachingProvider = {
    if (StringUtils.hasText(cachesProperties.providerConfiguration.provider)) {
      Caching.getCachingProvider(cachesProperties.providerConfiguration.provider)
    } else {
      Caching.getCachingProviders.asScala.toList.filterNot(provider => {
        provider.toString.contains(DEFAULT_EMBEDDED_CACHING_PROVIDER)
      }).head
    }
  }
}