package ai.digital.deploy.cache.service

import ai.digital.configuration.central.deploy.CacheConfiguration
import ai.digital.deploy.cache.config.CacheProvider
import grizzled.slf4j.Logging

import java.util
import javax.annotation.PostConstruct

trait CacheDataService[K, V] extends Logging {

  val cacheName: String

  val cacheProvider: CacheProvider

  val cacheConfiguration: CacheConfiguration

  private var cacheWrapper: CacheWrapper[K, V] = _

  @PostConstruct
  protected def registerCache(): Unit = {
    try {
      if (cacheProvider != null) {
        cacheWrapper = cacheProvider.registerCache[K, V](cacheName, cacheConfiguration)
        logger.info(s"$cacheName registered with cache manager")
      }
    } catch {
      case exception: Exception =>
        logger.error(s"Exception occurred while registering cache($cacheName), $exception")
    }
  }

  def containsKey(key: K): Boolean = {
    tryWithReturn(false) {
      cacheWrapper.containsKey(key)
    } { exception =>
      logger.error(s"Exception occurred while checking if the cache contains an entry for the specified key($key), " +
        s"$exception")
      false
    }
  }

  def get(key: K): Option[V] = {
    tryWithReturn[Option[V]](None) {
      Option(cacheWrapper.get(key))
    } { exception =>
      logger.error(s"Exception occurred while getting from cache with key($key), $exception")
      None
    }
  }

  def put(key: K, value: V): Unit = {
    tryWithoutReturn {
      cacheWrapper.put(key, value)
    } { exception =>
      logger.error(s"Exception occurred while putting into cache with key($key), $exception")
    }
  }

  def remove(key: K): Boolean = {
    tryWithReturn(false) {
      cacheWrapper.remove(key)
    } { exception =>
      logger.error(s"Exception occurred while removing from cache with key($key), $exception")
      false
    }
  }

  def getAll: Option[util.Map[K, V]] = {
    tryWithReturn[Option[util.Map[K, V]]](None) {
      Option(cacheWrapper.getAll)
    } { exception =>
      logger.error(s"Exception occurred while getting map of all cache entries, $exception")
      None
    }
  }

  def getAll(keys: util.Set[_ <: K]): Option[util.Map[K, V]] = {
    tryWithReturn[Option[util.Map[K, V]]](None) {
      Option(cacheWrapper.getAll(keys))
    } { exception =>
      logger.error(s"Exception occurred while getting map of cache entries with keys($keys), $exception")
      None
    }
  }

  def putAll(entries: util.Map[_ <: K, _ <: V]): Unit = {
    tryWithoutReturn {
      cacheWrapper.putAll(entries)
    } { exception =>
      logger.error(s"Exception occurred while putting entries with keys(" +
        s"${if (entries != null) entries.keySet() else null}) into cache, $exception")
    }
  }

  def removeAll(keys: util.Set[_ <: K]): Unit = {
    tryWithoutReturn {
      cacheWrapper.removeAll(keys)
    } { exception =>
      logger.error(s"Exception occurred while removing from cache with keys($keys), $exception")
    }
  }

  def removeAll(): Unit = {
    tryWithoutReturn {
      cacheWrapper.removeAll()
    } { exception =>
      logger.error(s"Exception occurred while removing all entries from cache, $exception")
    }
  }

  def clear(): Unit = {
    tryWithoutReturn {
      cacheWrapper.clear()
    } { exception =>
      logger.error(s"Exception occurred while clearing contents of the cache, $exception")
    }
  }

  private def tryWithoutReturn(block: => Unit)
                              (onExceptionCaught: Exception => Unit): Unit = {
    try {
      if (cacheWrapper != null) {
        block
      }
    } catch {
      case exception: Exception =>
        onExceptionCaught(exception)
    }
  }

  private def tryWithReturn[R](default: R)
                              (block: => R)
                              (onExceptionCaught: Exception => R): R = {
    try {
      if (cacheWrapper != null) {
        block
      } else {
        default
      }
    } catch {
      case exception: Exception =>
        onExceptionCaught(exception)
    }
  }

}
