package com.xebialabs.deployit.cli;

import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.kohsuke.args4j.spi.StopOptionHandler;

import java.io.File;
import java.io.PrintStream;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

public class CliOptions {

    static final int DEFAULT_DEPLOYIT_SERVER_PORT = 4516;
    static final int DEFAULT_DEPLOYIT_SERVER_SECURE_PORT = 4517;
    static final int DEFAULT_SO_TIMEOUT = 10000;
    static final String DEFAULT_DEPLOYIT_SERVER_HOST = "127.0.0.1";

    @Option(name = "-ext", usage = "Extension folder name, default is 'ext'")
    private String extensionFolderName = "ext";

    @Option(name = "-secure", usage = "Use https to connect to the XL Deploy server")
    private boolean secured = false;

    @Option(name = "-expose-proxies")
    private boolean exposeProxies = false;

    @Option(name = "-host", usage = "Connect to a specified host, defaults to " + DEFAULT_DEPLOYIT_SERVER_HOST)
    private String host = DEFAULT_DEPLOYIT_SERVER_HOST;

    @Option(name = "-port", usage = "Connect to a specified port, defaults to " + DEFAULT_DEPLOYIT_SERVER_PORT)
    private int port = -1;

    @Option(name = "-context", usage = "Context url XL Deploy is running at, defaults to ''")
    private String context = "";

    @Option(name = "-f", usage = "Execute a specified python source file or files (specified via comma)", aliases = {"-source"})
    private String sourceFileName = null;

    @Option(name = "-q", aliases = "-quiet", usage = "Suppresses the welcome banner")
    private boolean quiet = false;

    @Option(name = "-username", usage = "Connect as the specified user")
    private String username;

    @Option(name = "-password", usage = "Connect with the specified password")
    private String password;

    @Option(name = "-socketTimeout", usage = "Define the default socket timeout (SO_TIMEOUT) in milliseconds which is the timeout for waiting for data. Defaults to " + DEFAULT_SO_TIMEOUT)
    private int socketTimeout = DEFAULT_SO_TIMEOUT;

    @Option(name = "-proxyHost", usage = "Proxy host")
    private String proxyUrl;

    @Option(name = "-proxyPort", usage = "Proxy port")
    private int proxyPort = -1;

    @Option(name = "-configuration", usage = "Specify the location of the configuration directory. The CLI will look for a deployit.conf in this location.")
    private String configLocation = "conf";

    @Option(name = "-help", usage = "Prints this usage message")
    private boolean printUsage = false;

    @Argument
    @Option(name = "--", handler = StopOptionHandler.class)
    private List<String> scriptArgs = newArrayList();

    private static final String DEPLOYIT_CONF_FILE_NAME = "deployit.conf";

    public int getSocketTimeout() {
        return socketTimeout;
    }

    public void setSocketTimeout(int socketTimeout) {
        this.socketTimeout = socketTimeout;
    }

    public String getProxyHost() {
        return proxyUrl;
    }

    public int getProxyPort() {
        return proxyPort;
    }

    public void setProxyPort(int proxyPort) {
        this.proxyPort = proxyPort;
    }

    public void setProxyUrl(String proxyUrl) {
        this.proxyUrl = proxyUrl;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public String getHost() {
        return host;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public int getPort() {
        return port == -1 ? (secured ? DEFAULT_DEPLOYIT_SERVER_SECURE_PORT : DEFAULT_DEPLOYIT_SERVER_PORT) : port;
    }

    public void setSourceFileName(String sourceFileName) {
        this.sourceFileName = sourceFileName;
    }

    public String[] getSourceFileNames() {
        return sourceFileName.split(",");
    }

    public boolean interpretSourceFile() {
        return (this.sourceFileName != null);
    }

    public boolean isUsernameOnCommandline() {
        return this.username != null;
    }

    public boolean isPasswordOnCommandline() {
        return this.password != null;
    }

    public boolean isExposeProxies() {
        return exposeProxies;
    }

    public void setExposeProxies(boolean exposeProxies) {
        this.exposeProxies = exposeProxies;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(final String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(final String password) {
        this.password = password;
    }

    public String getConfigLocation() {
        return configLocation;
    }

    public void setConfigLocation(final String configLocation) {
        this.configLocation = configLocation;
    }

    public File getConfigurationFile() {
        return new File(configLocation, DEPLOYIT_CONF_FILE_NAME);
    }

    public boolean isSecured() {
        return secured;
    }

    public void setSecured(boolean secured) {
        this.secured = secured;
    }

    public String getUrl() {
        return format("http%s://%s:%d/%s", secured ? "s" : "", host, getPort(), getContext());
    }

    public List<String> getScriptArgs() {
        return scriptArgs;
    }

    public void setScriptArgs(List<String> scriptArgs) {
        this.scriptArgs = scriptArgs;
    }

    public String getExtensionFolderName() {
        return extensionFolderName;
    }

    public void setExtensionFolderName(String extensionFolderName) {
        this.extensionFolderName = extensionFolderName;
    }

    public String getContext() {
        if (!context.endsWith("deployit")) {
            context = context + "/deployit";
        }
        if (context.startsWith("/")) {
            context = context.substring(1);
        }
        return context;
    }



    public void setContext(String context) {
        this.context = context;
    }

    public boolean isPrintUsage() {
        return printUsage;
    }

    public void setPrintUsage(boolean printUsage) {
        this.printUsage = printUsage;
    }

    public boolean isQuiet() {
        return quiet;
    }

    public void setQuiet(boolean quiet) {
        this.quiet = quiet;
    }

    static CliOptions parse(String... args) {
        CliOptions options = new CliOptions();
        final CmdLineParser parser = new CmdLineParser(options);
        try {
            parser.parseArgument(args);

            if (options.isPrintUsage()) {
                printUsage(parser, System.err);
                return null;
            }
        } catch (CmdLineException e) {
            System.err.println(e.getMessage());
            printUsage(parser, System.out);
            return null;
        }

        return options;
    }

    private static void printUsage(CmdLineParser parser, PrintStream stream) {
        stream.println("Usage: ./cli." + (System.getProperty("os.name").matches("(?i).*windows.*") ? "cmd" : "sh") + " [options] [[--] arguments]\n");
        stream.println("Options:\n");
        parser.printUsage(stream);
    }
}
