package com.xebialabs.deployit.deployment.rules

import com.xebialabs.deployit.plugin.api.rules.Scope
import com.xebialabs.platform.script.jython._
import com.xebialabs.xlplatform.script.jython.JythonSugarDiscovery
import grizzled.slf4j.Logging

import scala.collection.mutable
import scala.util.Try

class SimpleRuleRegistry extends RuleRegistry with Logging {

  /**
   * Rule Registry
   */
  private[this] val rulesByScope: SortedMultiMap = new mutable.HashMap[Scope, mutable.Set[Rule]]() with SortedMultiMap

  private[this] val rulesByName = mutable.Map[String, Rule]()

  override def registerRule(rule: Rule): Unit = {
    if (isRuleNameRegistered(rule.getName)) {
      throw new IllegalArgumentException("Rule with duplicate name [%s] found." format rule.getName)
    }
    debug(s"Registering rule $rule")
    registerRuleByName(rule)
    registerRuleByScope(rule)
  }

  private def isRuleNameRegistered(name: String): Boolean = rulesByName.isDefinedAt(name)

  private def registerRuleByName(rule: Rule): Unit = rulesByName(rule.getName) = rule

  private def registerRuleByScope(rule: Rule) = rulesByScope.addBinding(rule.getScope, rule)

  override def getRule(name: String): Option[Rule] = rulesByName.get(name)

  override def getEnabledRules(scope: Scope): Iterable[Rule] = rulesByScope.getOrElse(scope, mutable.Set[Rule]()).filter(_.isEnabled)

  override def disableRule(name: String): Unit =
    getRule(name) match {
      case Some(rule) =>
        debug(s"Disabling rule [$name]")
        rule.disable()
      case _ => warn(s"Rule [$name] cannot be disable since it does not exist in the registry.")
    }

  def preloadScriptRuleSources(): Unit = {
    val dummyContext = Map(
      ContextFactory.DELTA -> null,
      ContextFactory.DEPLOYED -> null,
      ContextFactory.PREVIOUS_DEPLOYED -> null,
      ContextFactory.DELTAS -> List(),
      ContextFactory.SPECIFICATION -> null,
      ContextFactory.DEPLOYED_APPLICATION -> null,
      ContextFactory.PREVIOUS_DEPLOYED_APPLICATION -> null,
      ContextFactory.CONTEXT -> null,
      ContextFactory.INTERNAL_STEP_FACTORY -> null
    )
    implicit val jCtx: JythonContext =
      JythonContext(
        Seq(Syntactic.wrapperLib, ruleLib, Syntactic.loggerLib) ++ JythonSugarDiscovery.getExtensionResources,
        Bindings.xlDeployApiServices ++ dummyContext
      )

    rulesByName.values.foreach {
      case r: ScriptRule =>
        Try(r.evaluateExpression[Boolean]("True", expr => s"bool($expr)"))
        Try(r.executeScript(ScriptSource.byContent(r.getScript)))
      case _ =>
    }
  }

  private[this] trait SortedMultiMap extends mutable.MultiMap[Scope, Rule] {
    override protected def makeSet: mutable.TreeSet[Rule] = mutable.TreeSet()(Ordering.by(rule => rule.getOrder.getOrElse(Integer.MAX_VALUE) -> rule.getName))
  }

}
