package com.xebialabs.deployit.test.support;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.read.ListAppender;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasItem;

/**
 * Junit Rule that captures logback events for specified logger and level.
 * Debug switch will dump captured events to standard error.
 * <p>
 * Use it like this:
 *
 * <pre>
 * <code>
 *  &#64;Rule
 *  public LogbackEventsCapture scriptEvents = new LogbackEventsCapture("com.xebialabs.deployit.plugin.generic.scripts", Level.TRACE, false);
 *
 *  &#64;Test
 *  public void shouldObfuscatePassword() throws Exception {
 *      final String password = "Goodbye moon";
 *      scriptEvents.doesntContainMessageWith(password);
 *      scriptEvents.doesntContainMessageWith(encodeBase64String(password.getBytes()));
 *      scriptEvents.containsMessageWith("********");
 *      ...
 * </code>
 * </pre>
 *
 * @author ilx
 */
public class LogbackEventsCapture implements TestRule {
    private static List<Matcher<Iterable<? super ILoggingEvent>>> matchers = new ArrayList<>();
    private String loggerName;
    private Level level;
    private boolean debug;

    public LogbackEventsCapture(String loggerName) {
        this(loggerName, Level.TRACE, false);
    }

    public LogbackEventsCapture(String loggerName, Level level) {
        this(loggerName, level, false);
    }

    public LogbackEventsCapture(String loggerName, Level level, boolean debug) {
        this.loggerName = loggerName;
        this.level = level;
        this.debug = debug;
    }

    public void containsMessageWith(String expectedMsg) {
        assertThatEvents(hasItem(Matchers.<ILoggingEvent>hasProperty("message", containsString(expectedMsg))));
    }

    public void doesntContainMessageWith(String rawMsg) {
        assertThatEvents(not(hasItem(Matchers.<ILoggingEvent>hasProperty("message", containsString(rawMsg)))));
    }

    public void assertThatEvents(Matcher<Iterable<? super ILoggingEvent>> matcher) {
        matchers.add(matcher);
    }

    @Override
    public Statement apply(final Statement base, final Description description) {
        return new Statement() {
            @Override
            public void evaluate() throws Throwable {
                ListAppender<ILoggingEvent> appender = new ListAppender<>();
                Logger scriptsLogger = (Logger) LoggerFactory.getLogger(loggerName);
                scriptsLogger.setLevel(level);
                scriptsLogger.addAppender(appender);
                appender.start();

                try {
                    base.evaluate();
                } finally {
                    List<ILoggingEvent> events = appender.list;
                    if (!matchers.isEmpty()) {
                        if (debug) {
                            dumpEvents(events);
                        }
                        try {
                            for (Matcher<Iterable<? super ILoggingEvent>> matcher : matchers) {
                                assertThat(events, matcher);
                            }
                        } finally {
                            matchers.clear();
                            appender.stop();
                            events.clear();
                        }
                    }
                }
            }

            private void dumpEvents(List<ILoggingEvent> events) {
                StringBuilder sb = new StringBuilder();
                String NL = System.getProperty("line.separator");
                for (ILoggingEvent event : events) {
                    sb.append(event.getLevel()).append(":").append(event.getMessage()).append(NL);
                }
                System.err.print(sb.toString());
            }
        };
    }
}
