package ai.digital.deploy.permissions.client.local

import ai.digital.deploy.permissions.api.rest.dto.RoleWithGlobalPermissionsDto
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleConverter._
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleWithGlobalPermissionsConverter._
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{GlobalPermissionsServiceClient, PaginatedResponse}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.EmbeddedPermissionServiceProfile
import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.service.{GlobalPermissionService, RoleService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Profile
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Service

import java.util.UUID
import scala.jdk.CollectionConverters._

@ConditionalOnProperty(name = Array("xl.permission-service.enabled"), havingValue = "true", matchIfMissing = true)
@Service
@Profile(Array(EmbeddedPermissionServiceProfile))
@TransactionalPermissionService
class LocalGlobalPermissionsServiceClient(
  @Autowired globalPermissionService: GlobalPermissionService,
  @Autowired roleService: RoleService
) extends GlobalPermissionsServiceClient {
  override def addPermissions(roleName: String, permissions: List[String]): RoleWithGlobalPermissionsDto = {
    val (role, gp) = globalPermissionService.add(roleName, permissions)
    RoleWithGlobalPermissionsDto(role, gp)
  }

  override def removePermissions(roleName: String, permissions: List[String]): RoleWithGlobalPermissionsDto = {
    val (role, gp) = globalPermissionService.remove(roleName, permissions)
    RoleWithGlobalPermissionsDto(role, gp)
  }

  @ReadOnlyTransactionalPermissionService
  override def read(roleName: String): List[String] =
    globalPermissionService.read(roleName)

  @ReadOnlyTransactionalPermissionService
  override def read(rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithGlobalPermissionsDto] = {
    val rolePage = roleService.read(rolePattern, PageRequest.of(page - 1, size, SortOrder.getSort(order, field)))
    PaginatedResponse[RoleWithGlobalPermissionsDto](
      rolePage.getContent.asScala.toList.map { role =>
        RoleWithGlobalPermissionsDto(
          role,
          role.globalPermissions.asScala.toList
        )
      },
      rolePage.getTotalElements,
      page,
      size,
      rolePage.hasNext
    )
  }

  @ReadOnlyTransactionalPermissionService
  override def getGlobalPermissionsForRoleId(roleId: String): List[String] =
    globalPermissionService.getGlobalPermissionsForRoleId(UUID.fromString(roleId))

  @ReadOnlyTransactionalPermissionService
  override def getGlobalPermissionsForPrincipal(principalName: String): Set[String] =
    globalPermissionService.getGlobalPermissionsForPrincipal(principalName)

  @ReadOnlyTransactionalPermissionService
  override def getGlobalPermissionsForRoles(roleNames: List[String]): List[RoleWithGlobalPermissionsDto] =
    globalPermissionService
      .read(roleNames).groupBy(_.role).map {
        case (role, permissions) =>
          RoleWithGlobalPermissionsDto(role, permissions.map(_.permissionName))
      }.toList

  @ReadOnlyTransactionalPermissionService
  override def readByRolePattern(rolePattern: String): List[RoleWithGlobalPermissionsDto] =
    Option(rolePattern)
      .map(rp =>
        globalPermissionService
          .readByRolePattern(rp)
      ).getOrElse(globalPermissionService.read()).groupBy(_.role).map {
        case (role, permissions) =>
          RoleWithGlobalPermissionsDto(role, permissions.map(_.permissionName))
      }.toList
}
