package ai.digital.deploy.permissions.client.local

import ai.digital.deploy.permissions.api.rest.dto.converters.RoleConverter._
import ai.digital.deploy.permissions.api.rest.dto.{RoleDto, RoleWithPrincipalsDto}
import ai.digital.deploy.permissions.client.configuration.PermissionServiceEntityNotFoundError
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{PaginatedResponse, RoleServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.EmbeddedPermissionServiceProfile
import ai.digital.deploy.permissions.jpa.{ReadOnlyTransactionalPermissionService, TransactionalPermissionService}
import ai.digital.deploy.permissions.model.Role
import ai.digital.deploy.permissions.service.{RolePrincipalService, RoleService}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Profile
import org.springframework.data.domain.PageRequest
import org.springframework.stereotype.Component

import java.util.UUID
import scala.jdk.CollectionConverters._

@Component
@ConditionalOnProperty(name = Array("xl.permission-service.enabled"), havingValue = "true", matchIfMissing = true)
@Profile(Array(EmbeddedPermissionServiceProfile))
@TransactionalPermissionService
class LocalRoleServiceClient(
  @Autowired roleService: RoleService,
  @Autowired rolePrincipalService: RolePrincipalService
) extends RoleServiceClient {
  @ReadOnlyTransactionalPermissionService
  override def getAll: List[RoleDto] = roleService.readAll()

  override def create(name: String, principals: List[String]): RoleWithPrincipalsDto = {
    roleService.create(name)
    val (role, rolePrincipals) = rolePrincipalService.add(name, principals)
    RoleWithPrincipalsDto(role, rolePrincipals.map(_.principalName))
  }

  override def createOrUpdate(role: RoleDto): RoleDto =
    roleService.createOrUpdate(Role(role.id, role.name))

  override def update(originalRoleName: String,
                      updatedRoleName: String,
                      principalsToCreate: Set[String],
                      principalsToDelete: Set[String]
  ): RoleWithPrincipalsDto = {
    rolePrincipalService.edit(originalRoleName, principalsToCreate.toList, principalsToDelete.toList)
    val role = roleService
      .read(originalRoleName)
      .getOrElse(throw PermissionServiceEntityNotFoundError(s"Couldn't update role ${originalRoleName} - role entity not found"))
    role.name = updatedRoleName
    val updatedRole = roleService.update(role)
    RoleWithPrincipalsDto(updatedRole, updatedRole.principals.asScala.map(_.principalName).toList)
  }

  override def removeAllReferences(roleName: String): Unit =
    roleService.deleteAllRoleReferences(roleName)

  override def create(name: String): RoleDto =
    roleService.create(name)

  override def update(id: UUID, roleName: String): RoleDto =
    roleService.update(Role(id, roleName))

  override def rename(name: String, newName: String): RoleDto = roleService.rename(name, newName)

  override def delete(roleName: String): Unit = roleService.delete(roleName)

  override def deleteById(roleId: String): Unit = roleService.delete(UUID.fromString(roleId))

  @ReadOnlyTransactionalPermissionService
  override def readById(roleId: String): Option[RoleDto] =
    roleService.read(UUID.fromString(roleId))

  @ReadOnlyTransactionalPermissionService
  override def read(roleName: String): Option[RoleDto] = roleService.read(roleName)

  @ReadOnlyTransactionalPermissionService
  override def read(namePattern: String, page: Int, size: Int, order: SortOrder, field: String): PaginatedResponse[RoleDto] = {
    val rolePage = roleService.read(namePattern, PageRequest.of(page - 1, size, SortOrder.getSort(order, field)))
    PaginatedResponse[RoleDto](
      rolePage.getContent.asScala.toList,
      rolePage.getTotalElements,
      page,
      size,
      rolePage.hasNext
    )
  }

  @ReadOnlyTransactionalPermissionService
  override def readByRolePattern(namePattern: String): List[RoleDto] =
    roleService.readByRolePattern(namePattern)

  override def removeAll(): Unit =
    roleService.removeAll()

  @ReadOnlyTransactionalPermissionService
  override def roleExists(roleName: String): Boolean = roleService.read(roleName).isDefined

  @ReadOnlyTransactionalPermissionService
  override def countRoles(roleNamePattern: String): Long =
    roleService.count(roleNamePattern)
}
