package ai.digital.deploy.permissions.client.remote

import ai.digital.deploy.permissions.api.rest.dto.RoleWithPrincipalsDto
import ai.digital.deploy.permissions.api.rest.pagination.{Order, Paging}
import ai.digital.deploy.permissions.api.rest.v1.{PrincipalRolesPaths, RolePrincipalsPaths}
import ai.digital.deploy.permissions.client.util.SortOrder
import ai.digital.deploy.permissions.client.{PaginatedResponse, RolePrincipalsServiceClient}
import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.NotEmbeddedPermissionServiceProfile
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.annotation.Profile
import org.springframework.data.domain.Page
import org.springframework.http.{HttpEntity, HttpMethod}
import org.springframework.stereotype.Component
import org.springframework.web.client.RestTemplate
import org.springframework.web.util.UriComponentsBuilder
import org.springframework.core.ParameterizedTypeReference

import scala.jdk.CollectionConverters._

@Component
@Profile(Array(NotEmbeddedPermissionServiceProfile))
class RemoteRolePrincipalsServiceClient(@Qualifier("permissionServiceRestTemplate") restTemplate: RestTemplate)
    extends BaseRemoteClient
    with RolePrincipalsServiceClient {
  private val basePath = permissionServiceUrl + RolePrincipalsPaths.BASE_PATH
  private val addPrincipalsPath = basePath + RolePrincipalsPaths.ADD_PRINCIPALS_PATH
  private val removePrincipalsPath = basePath + RolePrincipalsPaths.REMOVE_PRINCIPALS_PATH
  private val readPrincipalsPath = basePath + RolePrincipalsPaths.READ_PRINCIPALS_PATH
  private val readPrincipalsAllRolesPath = basePath + RolePrincipalsPaths.READ_PRINCIPALS_PATH_ALL_ROLES

  private val principalRolesBasePath = permissionServiceUrl + PrincipalRolesPaths.BASE_PATH
  private val readRoles = principalRolesBasePath + PrincipalRolesPaths.READ_PRINCIPAL_ROLES_PATH
  private val readRolesForRolePatternPath = basePath + RolePrincipalsPaths.READ_PRINCIPALS_FOR_ROLE_PATTERN
  private val readRolesForPrincipalAndRolePatternPath =
    basePath + RolePrincipalsPaths.READ_PRINCIPALS_FOR_PRINCIPAL_AND_ROLE_PATTERN
  private val readRolesForPrincipalsAndRolePatternPath =
    basePath + RolePrincipalsPaths.READ_PRINCIPALS_FOR_PRINCIPALS_AND_ROLE_PATTERN
  private val readRolesForPrincipalsAndRolePatternWithPagingPath =
    basePath + RolePrincipalsPaths.READ_PRINCIPALS_FOR_PRINCIPALS_AND_ROLE_PATTERN_WITH_PAGING

  override def addPrincipals(roleName: String, principals: List[String]): RoleWithPrincipalsDto = {
    val uri = UriComponentsBuilder.fromUriString(addPrincipalsPath).build(roleName)
    restTemplate.postForObject(uri, principals, classOf[RoleWithPrincipalsDto])
  }

  override def removePrincipals(roleName: String, principals: List[String]): RoleWithPrincipalsDto = {
    val uri = UriComponentsBuilder.fromUriString(removePrincipalsPath).build(roleName)
    restTemplate
      .exchange(uri, HttpMethod.DELETE, new HttpEntity(principals), classOf[RoleWithPrincipalsDto]).getBody
  }

  override def readPrincipals(roleName: String): List[String] = {
    val responseType = new ParameterizedTypeReference[List[String]]() {}
    val uri = UriComponentsBuilder.fromUriString(readPrincipalsPath).build(roleName)
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def read(rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithPrincipalsDto] = {
    val responseType = new ParameterizedTypeReference[Array[RoleWithPrincipalsDto]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(readPrincipalsAllRolesPath)
      .queryParam("namePattern", rolePattern)
      .queryParam(Paging.PAGE_PARAMETER, page)
      .queryParam(Paging.SIZE_PARAMETER, size)
      .queryParam(Order.ORDER_PARAMETER, s"$field:${order.value}")
      .build().toUri
    val response = restTemplate.exchange(uri, HttpMethod.GET, null, responseType)
    val headers = response.getHeaders
    PaginatedResponse(
      response.getBody.toList,
      headers.getFirst(Paging.X_TOTAL_COUNT_HEADER).toLong,
      headers.getFirst(Paging.X_PAGE).toInt,
      headers.getFirst(Paging.X_SIZE).toInt,
      headers.getFirst(Paging.X_HAS_NEXT_PAGE_HEADER).toBoolean
    )
  }
  override def read(principal: String,
                    rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithPrincipalsDto] = {
    val responseType = new ParameterizedTypeReference[Array[RoleWithPrincipalsDto]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(readRoles)
      .queryParam("namePattern", rolePattern)
      .queryParam(Paging.PAGE_PARAMETER, page)
      .queryParam(Paging.SIZE_PARAMETER, size)
      .queryParam(Order.ORDER_PARAMETER, s"$field:${order.value}")
      .build(principal)
    val response = restTemplate.exchange(uri, HttpMethod.GET, null, responseType)
    val headers = response.getHeaders
    PaginatedResponse(
      response.getBody.toList,
      headers.getFirst(Paging.X_TOTAL_COUNT_HEADER).toLong,
      headers.getFirst(Paging.X_PAGE).toInt,
      headers.getFirst(Paging.X_SIZE).toInt,
      headers.getFirst(Paging.X_HAS_NEXT_PAGE_HEADER).toBoolean
    )
  }

  override def read(principals: List[String],
                    rolePattern: String,
                    page: Int,
                    size: Int,
                    order: SortOrder,
                    field: String
  ): PaginatedResponse[RoleWithPrincipalsDto] = {
    val responseType = new ParameterizedTypeReference[Array[RoleWithPrincipalsDto]]() {}
    val uri = UriComponentsBuilder
      .fromUriString(readRolesForPrincipalsAndRolePatternWithPagingPath)
      .queryParam("namePattern", rolePattern)
      .queryParam("principals", principals.asJava)
      .queryParam(Paging.PAGE_PARAMETER, page)
      .queryParam(Paging.SIZE_PARAMETER, size)
      .queryParam(Order.ORDER_PARAMETER, s"$field:${order.value}")
      .build().toUri
    val response = restTemplate.exchange(uri, HttpMethod.GET, null, responseType)
    val headers = response.getHeaders
    PaginatedResponse(
      response.getBody.toList,
      headers.getFirst(Paging.X_TOTAL_COUNT_HEADER).toLong,
      headers.getFirst(Paging.X_PAGE).toInt,
      headers.getFirst(Paging.X_SIZE).toInt,
      headers.getFirst(Paging.X_HAS_NEXT_PAGE_HEADER).toBoolean
    )
  }

  override def readByRolePattern(rolePattern: String): List[RoleWithPrincipalsDto] = {
    val responseType = new ParameterizedTypeReference[List[RoleWithPrincipalsDto]]() {}
    val uri =
      UriComponentsBuilder
        .fromUriString(readRolesForRolePatternPath)
        .queryParam(RolePrincipalsPaths.PARAM_ROLE_NAME_PATTERN, rolePattern).build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def read(principal: String, rolePattern: String): List[RoleWithPrincipalsDto] = {
    val responseType = new ParameterizedTypeReference[List[RoleWithPrincipalsDto]]() {}
    val uri =
      UriComponentsBuilder
        .fromUriString(readRolesForPrincipalAndRolePatternPath)
        .queryParam(RolePrincipalsPaths.PARAM_ROLE_NAME_PATTERN, rolePattern).build(principal)
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }

  override def read(principals: List[String], rolePattern: String): List[RoleWithPrincipalsDto] = {
    val responseType = new ParameterizedTypeReference[List[RoleWithPrincipalsDto]]() {}
    val uri =
      UriComponentsBuilder
        .fromUriString(readRolesForPrincipalsAndRolePatternPath)
        .queryParam(RolePrincipalsPaths.PARAM_ROLE_NAME_PATTERN, rolePattern)
        .queryParam("principals", principals.asJava)
        .build().toUri
    restTemplate.exchange(uri, HttpMethod.GET, null, responseType).getBody
  }
}
