package ai.digital.deploy.permissions.client.configuration

import ai.digital.deploy.permissions.config.profile.PermissionServiceProfileConfig.NotEmbeddedPermissionServiceProfile
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import org.apache.hc.client5.http.config.{ConnectionConfig, RequestConfig}
import org.apache.hc.client5.http.impl.auth.SystemDefaultCredentialsProvider
import org.apache.hc.client5.http.impl.classic.HttpClients
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManagerBuilder
import org.apache.hc.client5.http.impl.routing.SystemDefaultRoutePlanner
import org.apache.hc.client5.http.ssl.{NoopHostnameVerifier, SSLConnectionSocketFactoryBuilder}
import org.apache.hc.core5.ssl.SSLContextBuilder
import org.apache.hc.core5.util.Timeout
import org.springframework.boot.web.client.RestTemplateBuilder
import org.springframework.context.annotation.{Bean, Configuration, Profile}
import org.springframework.http.HttpStatus
import org.springframework.http.HttpStatus.Series
import org.springframework.http.client.{ClientHttpResponse, HttpComponentsClientHttpRequestFactory}
import org.springframework.web.client.{ResponseErrorHandler, RestTemplate}

import java.io.File
import java.net.ProxySelector
import java.security.cert.X509Certificate

@Configuration
@Profile(Array(NotEmbeddedPermissionServiceProfile))
class PermissionServiceClientConfiguration(serverConfiguration: ServerConfiguration) {

  @Bean
  def restTemplateResponseErrorHandler(): ResponseErrorHandler =
    new ResponseErrorHandler() {
      override def hasError(httpResponse: ClientHttpResponse): Boolean =
        HttpStatus.valueOf(httpResponse.getStatusCode.value()).series() == Series.CLIENT_ERROR ||
          HttpStatus.valueOf(httpResponse.getStatusCode.value()).series() == Series.SERVER_ERROR

      override def handleError(response: ClientHttpResponse): Unit =
        Series.valueOf(response.getStatusCode.value()) match {
          case Series.SERVER_ERROR =>
            throw PermissionServiceServerError(HttpStatus.valueOf(response.getStatusCode.value()).getReasonPhrase)
          case Series.CLIENT_ERROR if response.getStatusCode == HttpStatus.NOT_FOUND =>
            throw PermissionServiceEntityNotFoundError(HttpStatus.valueOf(response.getStatusCode.value()).getReasonPhrase)
          case Series.CLIENT_ERROR =>
            throw PermissionServiceClientError(HttpStatus.valueOf(response.getStatusCode.value()).getReasonPhrase)
        }

    }

  @Bean
  def permissionServiceRestTemplate(builder: RestTemplateBuilder): RestTemplate = {
    val sslContextBuilder = new SSLContextBuilder
    val httpClientBuilder = HttpClients.custom
    val timeout = Timeout.ofMinutes(1)

    if (serverConfiguration.isSsl)
      Option(serverConfiguration.getKeyStorePath)
        .map(path =>
          sslContextBuilder.loadTrustMaterial(new File(path),
                                              ServerConfigurationHelper.getKeyStorePassword.map(_.toCharArray).orNull
          )
        )
        .getOrElse(sslContextBuilder.loadTrustMaterial(null, (_: Array[X509Certificate], _: String) => true))
    else
      sslContextBuilder.loadTrustMaterial(null, (_: Array[X509Certificate], _: String) => true)
    httpClientBuilder.setRoutePlanner(new SystemDefaultRoutePlanner(ProxySelector.getDefault))
    httpClientBuilder.setDefaultCredentialsProvider(new SystemDefaultCredentialsProvider)

    val connectionManager = PoolingHttpClientConnectionManagerBuilder
      .create()
      .setSSLSocketFactory(
        SSLConnectionSocketFactoryBuilder
          .create()
          .setSslContext(sslContextBuilder.build)
          .setHostnameVerifier(new NoopHostnameVerifier)
          .build()
      )
      .setDefaultConnectionConfig(ConnectionConfig.custom().setConnectTimeout(timeout).build())
      .build()
    httpClientBuilder.setConnectionManager(connectionManager)
    httpClientBuilder
      .setDefaultRequestConfig(RequestConfig.custom.setResponseTimeout(timeout).build)
    builder
      .requestFactory(() => new HttpComponentsClientHttpRequestFactory(httpClientBuilder.build()))
      .errorHandler(restTemplateResponseErrorHandler())
      .build()
  }

}

final case class PermissionServiceServerError(message: String) extends DeployitException(message)
final case class PermissionServiceClientError(message: String) extends DeployitException(message)
final case class PermissionServiceEntityNotFoundError(message: String) extends DeployitException(message)
