package ai.digital.deploy.permissions.api.rest

import ai.digital.deploy.permissions.MessageHandler
import ai.digital.deploy.permissions.api.error.{PaginationParseError, ResourceNotFoundError, ValidationError}
import ai.digital.deploy.permissions.exception.{PaginationParseException, ResourceNotFoundException, RoleNotFoundException}
import org.springframework.http.HttpStatus
import org.springframework.validation.BindException
import org.springframework.web.bind.annotation.{ExceptionHandler, ResponseStatus, RestControllerAdvice}

import scala.jdk.CollectionConverters._

@RestControllerAdvice
class ErrorHandlingController(messageHandler: MessageHandler) {
  private val validationError: String = "error.validation"

  @ResponseStatus(HttpStatus.BAD_REQUEST)
  @ExceptionHandler(Array(classOf[BindException], classOf[PaginationParseException]))
  def handleValidationError(ex: Exception): AnyRef =
    ex match {
      case e: BindException =>
        ValidationError(
          messageHandler.getMessage(validationError),
          e.getBindingResult.getFieldErrors.asScala.toList
        )
      case e: PaginationParseException => PaginationParseError(e.getMessage)
    }

  @ResponseStatus(HttpStatus.NOT_FOUND)
  @ExceptionHandler(Array(classOf[ResourceNotFoundException], classOf[RoleNotFoundException]))
  def handleNotFound(ex: ResourceNotFoundException): ResourceNotFoundError =
    ex match {
      case e: RoleNotFoundException =>
        ResourceNotFoundError(messageHandler.getMessage("role.not.found", e.name))
      case e: ResourceNotFoundException => ResourceNotFoundError(e.message)
    }
}
