package ai.digital.deploy.permissions.api.rest

import ai.digital.deploy.permissions.MessageHandler
import ai.digital.deploy.permissions.api.error.{PaginationParseError, ResourceAlreadyExistsError, ResourceNotFoundError, ValidationError}
import ai.digital.deploy.permissions.exception._
import org.springframework.http.HttpStatus
import org.springframework.validation.{BindException, FieldError}
import org.springframework.web.bind.annotation.{ExceptionHandler, ResponseStatus, RestControllerAdvice}

import javax.validation.{ConstraintViolationException, ElementKind}
import scala.jdk.CollectionConverters._

@RestControllerAdvice
class ErrorHandlingController(messageHandler: MessageHandler) {
  private val validationError: String = "error.validation"

  @ResponseStatus(HttpStatus.BAD_REQUEST)
  @ExceptionHandler(Array(classOf[BindException], classOf[PaginationParseException], classOf[ConstraintViolationException]))
  def handleValidationError(ex: Exception): AnyRef =
    ex match {
      case e: BindException =>
        ValidationError(
          messageHandler.getMessage(validationError),
          e.getBindingResult.getFieldErrors.asScala.toList
        )
      case e: PaginationParseException => PaginationParseError(e.getMessage)
      case e: ConstraintViolationException =>
        ValidationError(
          messageHandler.getMessage(validationError),
          e.getConstraintViolations.asScala.toList.map(cv =>
            new FieldError(
              cv.getPropertyPath.asScala.toList.find(_.getKind == ElementKind.METHOD).getOrElse(throw e).getName,
              cv.getPropertyPath.asScala.toList.find(_.getKind == ElementKind.PARAMETER).getOrElse(throw e).getName,
              cv.getMessage
            )
          )
        )
    }

  @ResponseStatus(HttpStatus.NOT_FOUND)
  @ExceptionHandler(Array(classOf[ResourceNotFoundException]))
  def handleNotFound(ex: ResourceNotFoundException): ResourceNotFoundError =
    ex match {
      case e: RoleNameNotFoundException =>
        ResourceNotFoundError(messageHandler.getMessage("role.name.not.found", e.name))
      case e: RoleIdNotFoundException =>
        ResourceNotFoundError(messageHandler.getMessage("role.id.not.found", e.id))
      case e: ResourceNotFoundException => ResourceNotFoundError(e.message)
    }

  @ResponseStatus(HttpStatus.BAD_REQUEST)
  @ExceptionHandler(Array(classOf[ResourceAlreadyExistsException]))
  def handleAlreadyExists(ex: ResourceAlreadyExistsException): ResourceAlreadyExistsError =
    ex match {
      case e: RoleNameAlreadyExistsException =>
        ResourceAlreadyExistsError(messageHandler.getMessage("role.name.already.exists", e.name))
      case e: ResourceAlreadyExistsException => ResourceAlreadyExistsError(e.message)
    }
}
