package ai.digital.deploy.permissions.api.rest.v1.impl

import ai.digital.deploy.permissions.api.rest.dto._
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleConverter._
import ai.digital.deploy.permissions.api.rest.v1.RolesPaths._
import ai.digital.deploy.permissions.api.rest.v1.RolesResource
import ai.digital.deploy.permissions.exception.{RoleIdNotFoundException, RoleNameNotFoundException}
import ai.digital.deploy.permissions.model.Role
import ai.digital.deploy.permissions.service.RoleService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.ApplicationContext
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.web.bind.annotation._

import java.util.UUID
import javax.validation.Valid

@RestController
@RequestMapping(Array(BASE_PATH))
class RolesController(roleService: RoleService) extends RolesResource {
  @Autowired
  var context: ApplicationContext = _

  @GetMapping(Array(READ_ALL_ROLES))
  override def readAll(): List[RoleDto] =
    roleService.readAll()

  @PostMapping
  override def create(@Valid @RequestBody createRoleRequest: CreateRoleRequest): RoleDto =
    roleService.create(createRoleRequest.name)

  @PostMapping(Array(CREATE_OR_UPDATE_ROLE_PATH))
  override def createOrUpdate(@Valid @RequestBody updateRoleRequest: UpdateRoleRequest): RoleDto =
    roleService.createOrUpdate(Role(UUID.fromString(updateRoleRequest.id), updateRoleRequest.name))

  @DeleteMapping(Array(DELETE_ROLE_BY_NAME))
  override def deleteByName(@PathVariable name: String): Unit =
    roleService.delete(name)

  @DeleteMapping(Array(DELETE_ROLE_BY_ID))
  override def deleteById(@Valid roleIdRequest: RoleIdRequest): Unit =
    roleService.delete(UUID.fromString(roleIdRequest.roleUuid))

  @DeleteMapping(Array(DELETE_ALL_ROLE_REFS))
  override def deleteAllRoleRefs(@PathVariable name: String): Unit =
    roleService.deleteAllRoleReferences(name)

  @DeleteMapping(Array(DELETE_ALL_ROLES))
  override def deleteAll(): Unit =
    roleService.removeAll()

  @PutMapping
  override def update(@Valid @RequestBody updateRoleRequest: UpdateRoleRequest): RoleDto =
    roleService.update(Role(UUID.fromString(updateRoleRequest.id), updateRoleRequest.name))

  @GetMapping(Array(READ_ROLE))
  override def read(@PathVariable name: String): RoleDto =
    roleService.read(name).getOrElse(throw RoleNameNotFoundException(name))

  @GetMapping
  override def read(filter: RoleFilterDto, pageable: Pageable): Page[RoleDto] =
    roleService.read(filter.namePattern, pageable)

  @GetMapping(Array(READ_ROLE_BY_ID))
  override def readById(@Valid roleIdRequest: RoleIdRequest): RoleDto =
    roleService.read(UUID.fromString(roleIdRequest.roleUuid)).getOrElse(throw RoleIdNotFoundException(roleIdRequest.roleUuid))

  @PutMapping(Array(RENAME))
  override def rename(@PathVariable name: String, @Valid @RequestBody createRoleRequest: CreateRoleRequest): RoleDto =
    roleService.rename(name, createRoleRequest.name)

  @GetMapping(Array(COUNT_ROLES_BY_PATTERN))
  override def count(@PathVariable roleNamePattern: String): Long =
    roleService.count(roleNamePattern)

  @GetMapping(Array(READ_ROLES_BY_PATTERN))
  override def readByRolePattern(filter: RoleFilterDto): List[RoleDto] =
    roleService.readByRolePattern(filter.namePattern)
}
