package ai.digital.deploy.permissions.api.rest.history.impl

import ai.digital.deploy.permissions.api.rest.dto._
import ai.digital.deploy.permissions.api.rest.dto.converters.ReferencedPermissionTimeRangeConverter._
import ai.digital.deploy.permissions.api.rest.history.HistoricalReferencedPermissionResource
import ai.digital.deploy.permissions.api.rest.util.PathUtils
import ai.digital.deploy.permissions.exception.RoleNameNotFoundException
import ai.digital.deploy.permissions.model.view.ReferencedPermissionHistory
import ai.digital.deploy.permissions.service.{ReferencedPermissionHistoryService, RoleService}
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.web.bind.annotation._

import java.util.{UUID, List => JList}
import javax.validation.Valid
import scala.jdk.CollectionConverters._

object HistoricalReferencedPermissionController extends PathUtils {
  final override val BASE_PATH = "/history/permissions/refs"

  final val PARAM_ROLE_ID = "roleId"
  final val PARAM_PRINCIPAL_NAME = "principalName"
  final val PARAM_REFERENCE = "reference"
  final val PARAM_PERMISSIONS = "permissions"
  final val PARAM_REFERENCED_PERMISSION_ID = "referencedPermissionId"
  final val PARAM_ROLE_NAME = "roleName"
  final val PARAM_PERMISSION_NAME = "permissionName"
  final val PARAM_START_TIME = "startTime"
  final val PARAM_END_TIME = "endTime"
}

import ai.digital.deploy.permissions.api.rest.history.impl.HistoricalReferencedPermissionController._

@RestController
@RequestMapping(Array(BASE_PATH))
class HistoricalReferencedPermissionController(referencedPermissionHistoryService: ReferencedPermissionHistoryService,
                                               roleService: RoleService
) extends HistoricalReferencedPermissionResource {
  @GetMapping(params = Array(PARAM_ROLE_ID, PARAM_PERMISSIONS, PARAM_START_TIME, PARAM_END_TIME))
  def listReferencesByRoleAndPermissions(
    @Valid request: GetReferencedPermissionTimeRangeByRole
  ): JList[ReferencedPermissionTimeRangeDto] =
    referencedPermissionHistoryService
      .listReferencesByRoleAndPermissions(UUID.fromString(request.roleId),
                                          request.permissions.asScala.toList,
                                          request.startTime,
                                          request.endTime
      )

  @GetMapping(params = Array(PARAM_PRINCIPAL_NAME, PARAM_PERMISSIONS, PARAM_START_TIME, PARAM_END_TIME))
  def listReferencesByPrincipalAndPermissions(
    @Valid request: GetReferencedPermissionTimeRangeByPrincipal
  ): JList[ReferencedPermissionTimeRangeDto] =
    referencedPermissionHistoryService
      .listReferencesByPrincipalAndPermissions(request.principalName,
                                               request.permissions.asScala.toList,
                                               request.startTime,
                                               request.endTime
      )

  @GetMapping(params = Array(PARAM_ROLE_ID, PARAM_PERMISSIONS, PARAM_REFERENCE, PARAM_START_TIME, PARAM_END_TIME))
  def listReferencesByRoleAndReferenceAndPermissions(
    @Valid request: GetReferencedPermissionTimeRangeByRoleAndRef
  ): JList[ReferencedPermissionTimeRangeDto] =
    referencedPermissionHistoryService
      .listReferencesByRoleAndReferenceAndPermissions(UUID.fromString(request.roleId),
                                                      UUID.fromString(request.reference),
                                                      request.permissions.asScala.toList,
                                                      request.startTime,
                                                      request.endTime
      )

  @GetMapping(params = Array(PARAM_PRINCIPAL_NAME, PARAM_PERMISSIONS, PARAM_REFERENCE, PARAM_START_TIME, PARAM_END_TIME))
  def listReferencesByPrincipalAndReferenceAndPermissions(
    @Valid request: GetReferencedPermissionTimeRangeByPrincipalAndRef
  ): JList[ReferencedPermissionTimeRangeDto] =
    referencedPermissionHistoryService
      .listReferencesByPrincipalAndReferenceAndPermissions(request.principalName,
                                                           UUID.fromString(request.reference),
                                                           request.permissions.asScala.toList,
                                                           request.startTime,
                                                           request.endTime
      )

  @GetMapping(params = Array(PARAM_PERMISSION_NAME, PARAM_REFERENCE, PARAM_START_TIME, PARAM_END_TIME))
  override def getReferenceHistoriesForReferenceAndPermission(@Valid dto: GetReferenceHistoriesByPermissionName,
                                                              pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    referencedPermissionHistoryService.getReferenceHistoriesForReferenceAndPermission(UUID.fromString(dto.reference),
                                                                                      dto.permissionName,
                                                                                      dto.startTime,
                                                                                      dto.endTime,
                                                                                      pageable
    )

  @GetMapping(params = Array(PARAM_ROLE_ID))
  override def getReferenceHistoriesForRoleId(@Valid dto: GetReferenceHistoriesByRoleId,
                                              pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    referencedPermissionHistoryService.getReferenceHistoriesForRoleId(UUID.fromString(dto.roleId), pageable)

  @GetMapping(params = Array(PARAM_ROLE_NAME))
  override def getReferenceHistoriesForRoleName(@Valid dto: GetReferenceHistoriesRoleName,
                                                pageable: Pageable
  ): Page[ReferencedPermissionHistory] = {
    val role = roleService
      .read(dto.roleName).getOrElse(throw RoleNameNotFoundException(dto.roleName))
    referencedPermissionHistoryService.getReferenceHistoriesForRoleId(role.id, pageable)
  }

  @GetMapping(params = Array(PARAM_ROLE_ID, PARAM_REFERENCE, PARAM_START_TIME, PARAM_END_TIME))
  override def getReferenceHistoriesForReferenceAndRoleId(@Valid dto: GetReferenceHistoriesByRoleIdTimeRange,
                                                          pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    referencedPermissionHistoryService.getReferenceHistoriesForReferenceAndRoleId(UUID.fromString(dto.reference),
                                                                                  UUID.fromString(dto.roleId),
                                                                                  dto.startTime,
                                                                                  dto.endTime,
                                                                                  pageable
    )

  @GetMapping(params = Array(PARAM_REFERENCED_PERMISSION_ID, PARAM_REFERENCE, PARAM_START_TIME, PARAM_END_TIME))
  override def getReferenceHistoriesForReferenceAndReferencedPermissionId(
    @Valid dto: GetReferenceHistoriesByReferencedPermissionId,
    pageable: Pageable
  ): Page[ReferencedPermissionHistory] =
    referencedPermissionHistoryService.getReferenceHistoriesForReferenceAndReferencedPermissionId(
      UUID.fromString(dto.reference),
      UUID.fromString(dto.referencedPermissionId),
      dto.startTime,
      dto.endTime,
      pageable
    )

}
