package ai.digital.deploy.permissions.api.rest.v1.impl

import ai.digital.deploy.permissions.api.rest.dto._
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleConverter._
import ai.digital.deploy.permissions.api.rest.v1.RolesPaths._
import ai.digital.deploy.permissions.exception.{RoleIdNotFoundException, RoleNameNotFoundException}
import ai.digital.deploy.permissions.model.Role
import ai.digital.deploy.permissions.service.RoleService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.ApplicationContext
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.web.bind.annotation._

import java.util.UUID
import javax.validation.Valid

@RestController
@RequestMapping(Array(BASE_PATH))
class RolesController(roleService: RoleService) {
  @Autowired
  var context: ApplicationContext = _

  @GetMapping(Array(READ_ALL_ROLES))
  def readAll(): List[RoleDto] =
    roleService.readAll()

  @PostMapping
  def create(@Valid @RequestBody createRoleRequest: CreateRoleRequest): RoleDto =
    roleService.create(createRoleRequest.name)

  @PostMapping(Array(CREATE_OR_UPDATE_ROLE_PATH))
  def createOrUpdate(@Valid @RequestBody updateRoleRequest: UpdateRoleRequest): RoleDto =
    roleService.createOrUpdate(Role(UUID.fromString(updateRoleRequest.id), updateRoleRequest.name))

  @DeleteMapping(Array(DELETE_ROLE_BY_NAME))
  def deleteByName(@PathVariable name: String): Unit =
    roleService.delete(name)

  @DeleteMapping(Array(DELETE_ROLE_BY_ID))
  def deleteById(@Valid roleIdRequest: RoleIdRequest): Unit =
    roleService.delete(UUID.fromString(roleIdRequest.roleUuid))

  @DeleteMapping(Array(DELETE_ALL_ROLE_REFS))
  def deleteAllRoleRefs(@PathVariable name: String): Unit =
    roleService.deleteAllRoleReferences(name)

  @DeleteMapping(Array(DELETE_ALL_ROLES))
  def deleteAll(): Unit =
    roleService.removeAll()

  @PutMapping
  def update(@Valid @RequestBody updateRoleRequest: UpdateRoleRequest): RoleDto =
    roleService.update(Role(UUID.fromString(updateRoleRequest.id), updateRoleRequest.name))

  @GetMapping(Array(READ_ROLE))
  def read(@PathVariable name: String): RoleDto =
    roleService.read(name).getOrElse(throw RoleNameNotFoundException(name))

  @GetMapping
  def read(filter: RoleFilterDto, pageable: Pageable): Page[RoleDto] =
    roleService.read(filter.namePattern, pageable)

  @GetMapping(Array(READ_ROLE_BY_ID))
  def readById(@Valid roleIdRequest: RoleIdRequest): RoleDto =
    roleService.read(UUID.fromString(roleIdRequest.roleUuid)).getOrElse(throw RoleIdNotFoundException(roleIdRequest.roleUuid))

  @PutMapping(Array(RENAME))
  def rename(@PathVariable name: String, @Valid @RequestBody createRoleRequest: CreateRoleRequest): RoleDto =
    roleService.rename(name, createRoleRequest.name)

  @GetMapping(Array(COUNT_ROLES_BY_PATTERN))
  def count(@PathVariable roleNamePattern: String): Long =
    roleService.count(roleNamePattern)

  @GetMapping(Array(READ_ROLES_BY_PATTERN))
  def readByRolePattern(filter: RoleFilterDto): List[RoleDto] =
    roleService.readByRolePattern(filter.namePattern)
}
