package ai.digital.deploy.permissions.api.rest.v1.impl

import ai.digital.deploy.permissions.api.rest.dto.converters.RoleConverter._
import ai.digital.deploy.permissions.api.rest.dto.converters.RoleWithGlobalPermissionsConverter._
import ai.digital.deploy.permissions.api.rest.dto.{CheckPermissionRequest, CheckPermissionWithPrincipalsRequest, CheckPermissionWithReferencesAndPrincipalsRequest, CheckPermissionWithReferencesRequest, ReferencedPermissionsDto, RoleFilterDto, RoleWithPermissionsDto}
import ai.digital.deploy.permissions.api.rest.v1.PermissionsPaths.{BASE_PATH, CHECK_PERMISSION_FOR_ROLES_AND_REFERENCE, CHECK_PERMISSION_FOR_ROLES_AND_REFERENCES, CHECK_PERMISSION_FOR_ROLES_REFERENCES_AND_PRINCIPALS, CHECK_PERMISSION_FOR_ROLES_REFERENCE_AND_PRINCIPALS, GET_ALL_PERMISSION_FOR_ROLE, GET_ALL_PERMISSION_FOR_ROLES, GET_ALL_PERMISSION_FOR_ROLES_WITH_FILTER}
import ai.digital.deploy.permissions.exception.RoleNameNotFoundException
import ai.digital.deploy.permissions.service.{GlobalPermissionService, ReferencedPermissionService, RoleService}
import org.springframework.data.domain.{Page, PageImpl, Pageable}
import org.springframework.web.bind.annotation._

import java.util.{Collections, UUID, List => JList}
import javax.validation.Valid
import scala.jdk.CollectionConverters._

@RestController
@RequestMapping(Array(BASE_PATH))
class PermissionController(roleService: RoleService,
                           referencedPermissionService: ReferencedPermissionService,
                           globalPermissionService: GlobalPermissionService
) {
  @GetMapping(Array(GET_ALL_PERMISSION_FOR_ROLE))
  def read(@PathVariable roleName: String): RoleWithPermissionsDto = {
    val globalPermissions = globalPermissionService.read(roleName).map(_.permissionName)
    val referencedPermissions = referencedPermissionService.read(roleName).groupBy(_.reference)
    val role = roleService.read(roleName).getOrElse(throw RoleNameNotFoundException(roleName))
    val referencedPermissionsDto = referencedPermissions.map { rp =>
      ReferencedPermissionsDto(rp._1, rp._2.map(_.permissionName))
    }.toList
    RoleWithPermissionsDto(role, globalPermissions, referencedPermissionsDto)
  }

  @GetMapping(Array(GET_ALL_PERMISSION_FOR_ROLES))
  def read(@RequestParam roleNames: JList[String]): List[RoleWithPermissionsDto] = {
    val globalPermissions = globalPermissionService.read(roleNames.asScala.toList).groupBy(_.role.name)
    val referencedPermissions = referencedPermissionService.read(roleNames.asScala.toList).groupBy(_.role.name)
    val roles = roleService.read(roleNames.asScala.toList)
    roles.map { role =>
      val referencedPermissionsForRole = referencedPermissions.getOrElse(role.name, List.empty).groupBy(_.reference)
      val referencedPermissionsDto = referencedPermissionsForRole.map { rp =>
        ReferencedPermissionsDto(rp._1, rp._2.map(_.permissionName))
      }.toList
      RoleWithPermissionsDto(role, globalPermissions.getOrElse(role.name, List.empty), referencedPermissionsDto)
    }
  }

  @GetMapping(Array(GET_ALL_PERMISSION_FOR_ROLES_WITH_FILTER))
  def read(@RequestParam roleNames: JList[String], pageable: Pageable): Page[RoleWithPermissionsDto] = {
    val roles = roleService.read(roleNames.asScala.toList, pageable)
    val names = roles.asScala.toList.map(_.name)
    val globalPermissions = globalPermissionService.read(names).groupBy(_.role.name)
    val referencedPermissions = referencedPermissionService.read(names).groupBy(_.role.name)
    val dtos = roles.asScala.toList.map { role =>
      val referencedPermissionsForRole = referencedPermissions.getOrElse(role.name, List.empty).groupBy(_.reference)
      val referencedPermissionsDto = referencedPermissionsForRole.map { rp =>
        ReferencedPermissionsDto(rp._1, rp._2.map(_.permissionName))
      }.toList
      RoleWithPermissionsDto(role, globalPermissions.getOrElse(role.name, List.empty), referencedPermissionsDto)
    }
    new PageImpl[RoleWithPermissionsDto](dtos.asJava, pageable, roles.getTotalElements)
  }

  @GetMapping(Array(CHECK_PERMISSION_FOR_ROLES_AND_REFERENCE))
  def checkPermission(checkPermissionRequest: CheckPermissionRequest): Boolean = {
    val referenceMaybe = Option[UUID](checkPermissionRequest.referenceMaybe)
    referenceMaybe
      .map { referenceId =>
        referencedPermissionService.checkPermission(referenceId,
                                                    checkPermissionRequest.permissions.asScala.toList,
                                                    checkPermissionRequest.allRoles.asScala.toList
        )
      }
      .getOrElse(
        globalPermissionService.checkPermission(checkPermissionRequest.permissions.asScala.toList,
                                                checkPermissionRequest.allRoles.asScala.toList
        )
      )
  }
  @GetMapping(Array(CHECK_PERMISSION_FOR_ROLES_REFERENCE_AND_PRINCIPALS))
  def checkPermission(
    checkPermissionRequest: CheckPermissionWithPrincipalsRequest
  ): Boolean = {
    val referenceMaybe = Option[UUID](checkPermissionRequest.referenceMaybe)
    referenceMaybe
      .map { referenceId =>
        referencedPermissionService.checkPermission(
          referenceId,
          checkPermissionRequest.permissions.asScala.toList,
          checkPermissionRequest.allRoles.asScala.toList,
          checkPermissionRequest.principals.asScala.toList
        )
      }
      .getOrElse(
        globalPermissionService
          .checkPermission(checkPermissionRequest.permissions.asScala.toList,
                           checkPermissionRequest.allRoles.asScala.toList,
                           checkPermissionRequest.principals.asScala.toList
          )
      )
  }

  @GetMapping(Array(CHECK_PERMISSION_FOR_ROLES_AND_REFERENCES))
  def checkPermission(
    checkPermissionRequest: CheckPermissionWithReferencesRequest
  ): Map[String, Boolean] =
    if (checkPermissionRequest.references.asScala.toList.isEmpty)
      Map(
        "global" -> globalPermissionService.checkPermission(checkPermissionRequest.permissions.asScala.toList,
                                                            checkPermissionRequest.allRoles.asScala.toList
        )
      )
    else
      checkPermissionRequest.references.asScala.toList
        .map(reference =>
          reference.toString -> referencedPermissionService
            .checkPermission(reference,
                             checkPermissionRequest.permissions.asScala.toList,
                             checkPermissionRequest.allRoles.asScala.toList
            )
        ).toMap

  @GetMapping(Array(CHECK_PERMISSION_FOR_ROLES_REFERENCES_AND_PRINCIPALS))
  def checkPermission(
    checkPermissionRequest: CheckPermissionWithReferencesAndPrincipalsRequest
  ): Map[String, Boolean] =
    if (checkPermissionRequest.references.asScala.toList.isEmpty)
      Map(
        "global" -> globalPermissionService.checkPermission(checkPermissionRequest.permissions.asScala.toList,
                                                            checkPermissionRequest.allRoles.asScala.toList,
                                                            checkPermissionRequest.principals.asScala.toList
        )
      )
    else
      checkPermissionRequest.references.asScala.toList
        .map(reference =>
          reference.toString -> referencedPermissionService.checkPermission(
            reference,
            checkPermissionRequest.permissions.asScala.toList,
            checkPermissionRequest.allRoles.asScala.toList,
            checkPermissionRequest.principals.asScala.toList
          )
        ).toMap
}
