package ai.digital.deploy.permissions.api.rest.dto.converters

import ai.digital.deploy.permissions.api.rest.dto.{RoleDto, RoleWithPrincipalsDto, RoleWithReferencedPermissionsDto}
import ai.digital.deploy.permissions.model.view.{RolePrincipalReferenceView, RoleReferencedPermissionView}
import ai.digital.deploy.permissions.model.{GlobalPermission, ReferencedPermission, Role, RolePrincipal}
import org.springframework.data.domain.{Page, PageImpl, Pageable}

import java.util.UUID
import scala.jdk.CollectionConverters._

object RoleConverter {

  import scala.language.implicitConversions

  implicit def roleToDto(role: Option[Role]): Option[RoleDto] = role.map(r => RoleDto(r.id, r.name))
  implicit def roleToDto(role: Role): RoleDto = RoleDto(role.id, role.name)

  implicit def roleToDto(roles: List[Role]): List[RoleDto] = roles.map(roleToDto)

  implicit def roleToDto(roles: Page[Role]): Page[RoleDto] =
    new PageImpl(
      roles.getContent.asScala.toList.map(roleToDto).asJava,
      roles.getPageable,
      roles.getTotalElements
    )

  implicit def rolePrincipalToDto(rolePrincipal: RolePrincipal): RoleDto = RoleDto(rolePrincipal.role.id, rolePrincipal.role.name)

  implicit def rolePrincipalToDto(rolePrincipals: List[RolePrincipal]): List[RoleDto] = rolePrincipals.map(rolePrincipalToDto)

  implicit def rolePrincipalToDto(rolePrincipals: Page[RolePrincipal]): Page[RoleDto] =
    new PageImpl(
      rolePrincipals.getContent.asScala.toList.map(rolePrincipalToDto).asJava,
      rolePrincipals.getPageable,
      rolePrincipals.getTotalElements
    )

  implicit def rolePrincipalToRoleWithPrincipalsDto(rolePrincipal: RolePrincipal): RoleWithPrincipalsDto =
    RoleWithPrincipalsDto(roleToDto(rolePrincipal.role), List(rolePrincipal.principalName))

  implicit def rolePrincipalToRoleWithPrincipalsDto(rolePrincipals: List[RolePrincipal]): List[RoleWithPrincipalsDto] = {
    val roles = rolePrincipals.map(_.role).distinct
    roles.map { role =>
      RoleWithPrincipalsDto(RoleDto(role.id, role.name),
                            rolePrincipals.filter(r => r.role.id.equals(role.id)).map(_.principalName)
      )
    }
  }

  implicit def rolePrincipalToRoleWithPrincipalsDto(rolePrincipals: Page[RolePrincipal]): Page[RoleWithPrincipalsDto] = {
    val roleWithPrincipalsDtoList = createRoleWithPrincipalsDtoListFromRolePrincipal(rolePrincipals.getContent.asScala.toList)
    new PageImpl(
      roleWithPrincipalsDtoList.asJava,
      rolePrincipals.getPageable,
      roleWithPrincipalsDtoList.size
    )
  }

  implicit def rolePrincipalReferenceToRoleWithPrincipalsDto(
    rolePrincipalsReference: Page[RolePrincipalReferenceView]
  ): Page[RoleWithPrincipalsDto] = {
    val roleWithPrincipalsDtoList = createRoleWithPrincipalsDtoListFromRolePrincipalReferenceView(
      rolePrincipalsReference.getContent.asScala.toList
    )
    new PageImpl(
      roleWithPrincipalsDtoList.asJava,
      rolePrincipalsReference.getPageable,
      roleWithPrincipalsDtoList.size
    )
  }

  private def createRoleWithPrincipalsDtoListFromRolePrincipal(rp: List[RolePrincipal]): List[RoleWithPrincipalsDto] = {
    val roles = rp.map(_.role).distinct
    roles.map { role =>
      RoleWithPrincipalsDto(roleToDto(role), rp.filter(_.role.equals(role)).map(_.principalName))
    }
  }

  private def createRoleWithPrincipalsDtoListFromRolePrincipalReferenceView(
    rp: List[RolePrincipalReferenceView]
  ): List[RoleWithPrincipalsDto] = {
    val roles = rp.map(r => (r.roleId, r.roleName)).distinct
    roles.map { role =>
      RoleWithPrincipalsDto(RoleDto(role._1, role._2), rp.filter(r => r.roleId.equals(role._1)).map(_.principalName))
    }
  }
}

object RoleWithPrincipalsConverter {
  import scala.language.implicitConversions

  implicit def rolePrincipalToDto(principal: RolePrincipal): String = principal.principalName
  implicit def rolePrincipalToDto(principals: List[RolePrincipal]): List[String] = principals.map(rolePrincipalToDto)
  implicit def roleWithPrincipalsToDto(
    pageable: Pageable,
    pageView: (Page[Role], List[RolePrincipal])
  ): Page[RoleWithPrincipalsDto] = {
    val (rolePage, views) = pageView
    val groupedViews = views.groupBy(_.role.id)
    val dtos = rolePage.getContent.asScala.toList.map { role =>
      RoleWithPrincipalsDto(RoleConverter.roleToDto(role), groupedViews.getOrElse(role.id, List.empty).map(_.principalName))
    }
    new PageImpl[RoleWithPrincipalsDto](dtos.asJava, pageable, rolePage.getTotalElements)
  }

  implicit def roleWithPrincipalsToDto(
    view: (List[Role], List[RolePrincipal])
  ): List[RoleWithPrincipalsDto] = {
    val (roles, views) = view
    val groupedViews = views.groupBy(_.role.id)
    roles.map { role =>
      RoleWithPrincipalsDto(RoleConverter.roleToDto(role), groupedViews.getOrElse(role.id, List.empty).map(_.principalName))
    }
  }

  implicit def roleWithPrincipalsReferenceViewToDto(
    pageable: Pageable,
    pageView: (Page[Role], List[RolePrincipalReferenceView])
  ): Page[RoleWithPrincipalsDto] = {
    val (rolePage, views) = pageView
    val groupedViews = views.groupBy(_.roleId)
    val dtos = rolePage.getContent.asScala.toList.map { role =>
      RoleWithPrincipalsDto(RoleConverter.roleToDto(role), groupedViews.getOrElse(role.id, List.empty).map(_.principalName))
    }
    new PageImpl[RoleWithPrincipalsDto](dtos.asJava, pageable, rolePage.getTotalElements)
  }
}

object RoleWithGlobalPermissionsConverter {
  import scala.language.implicitConversions

  implicit def globalPermissionToDto(permission: GlobalPermission): String = permission.permissionName
  implicit def globalPermissionToDto(permissions: List[GlobalPermission]): List[String] = permissions.map(globalPermissionToDto)
}

object RoleWithReferencedPermissionsConverter {
  import RoleConverter._

  import scala.language.implicitConversions

  implicit def permissionToDto(permission: ReferencedPermission): String = permission.permissionName
  implicit def permissionToDto(permissions: List[ReferencedPermission]): List[String] = permissions.map(permissionToDto)

  implicit def roleReferencedPermissionViewToDto(
    referenceId: UUID,
    pageable: Pageable,
    pageView: (Page[Role], List[RoleReferencedPermissionView])
  ): Page[RoleWithReferencedPermissionsDto] = {
    val (rolePage, views) = pageView
    val groupedViews = views.groupBy(_.roleId)
    val dtos = rolePage.getContent.asScala.toList
      .map { role =>
        RoleWithReferencedPermissionsDto(role, referenceId, groupedViews.getOrElse(role.id, List.empty).map(_.permissionName))
      }
    new PageImpl[RoleWithReferencedPermissionsDto](dtos.asJava, pageable, rolePage.getTotalElements)
  }

}
