package ai.digital.deploy.gitops;

/**
 * Service interface for Git repository operations supporting DevOps-as-Code workflows.
 * Provides comprehensive Git operations including connection testing, cloning, pulling,
 * and repository validation for XL Deploy GitOps plugin.
 */
public interface GitOpsService {
    
    /**
     * Tests connection to a Git repository by attempting to list remote branches.
     * Validates both connectivity and branch existence if specified.
     * 
     * @param url the Git repository URL (HTTP/HTTPS/SSH)
     * @param branch the branch name to validate (can be null for default branch)
     * @param credentials the Git credentials for authentication (can be null for public repos)
     * @throws RuntimeException if connection fails or branch doesn't exist
     */
    void testConnection(String url, String branch, GitUserCredentials credentials);
    
    /**
     * Clones a Git repository to the specified target directory.
     * Creates org-prefixed directory names and handles collision resolution.
     * 
     * @param url the Git repository URL to clone from
     * @param branch the specific branch to clone (null for default branch)
     * @param credentials the Git credentials for authentication (can be null for public repos)
     * @param targetDirectory the base directory where repository will be cloned
     * @return the absolute path to the cloned repository directory
     * @throws RuntimeException if clone operation fails
     */
    String cloneRepository(String url, String branch, GitUserCredentials credentials, String targetDirectory);
    
    /**
     * Clones a Git repository using XL Deploy ConfigurationItem as the source.
     * Extracts URL, branch, and credentials from the configuration item.
     * 
     * @param gitSource the XL Deploy ConfigurationItem containing Git repository configuration
     * @param targetDirectory the base directory where repository will be cloned
     * @return the absolute path to the cloned repository directory
     * @throws IllegalArgumentException if gitSource is null
     * @throws RuntimeException if clone operation fails
     */
    String cloneRepository(com.xebialabs.deployit.plugin.api.udm.ConfigurationItem gitSource, String targetDirectory);
    
    /**
     * Updates an existing Git repository by pulling latest changes from remote.
     * Handles merge conflicts automatically using configurable resolution strategies.
     * 
     * @param repositoryPath the absolute path to the existing Git repository
     * @param credentials the Git credentials for authentication (can be null for public repos)
     * @return the repository path (same as input if successful)
     * @throws RuntimeException if pull operation fails or repository is invalid
     */
    String pullRepository(String repositoryPath, GitUserCredentials credentials);
    
    /**
     * Checks if the specified directory contains a valid Git repository.
     * Validates the presence of .git directory and basic repository structure.
     * 
     * @param directoryPath the absolute path to check
     * @return true if directory is a valid Git repository, false otherwise
     */
    boolean isGitRepository(String directoryPath);
    
    /**
     * Determines the actual repository path that would be used for cloning operations.
     * Takes into account org-prefixed naming and collision resolution logic.
     * 
     * @param url the Git repository URL
     * @param targetDirectory the base target directory
     * @return the calculated absolute path where repository would be cloned
     */
    String determineRepositoryPath(String url, String targetDirectory);
}